import {NotificationSettings} from '../notification-settings';

export default class FolderNotificationsPage extends NotificationSettings {

    constructor() {
        super();
        cy.get('.folder-notifications-page').should('be.visible');
    }

    getInheritParagraph() {
        return cy.get(`p.inherit-checkbox`);
    }

    getInheritAnchorTag() {
        return this.getInheritParagraph().find('a');
    }

    getInheritCheckbox() {
        return this.getInheritParagraph().find('input[type=checkbox]');
    }

    getSaveButton() {
        return cy.get(".header button.primary:contains('Save')");
    }

    getResetButton() {
        return cy.get(".header button:contains('Reset')");
    }

    getLastSavedTimeElement() {
        return cy.get("header .last-saved:contains('Last saved on')");
    }

    getInheritConfirmationModalElement() {
        return cy.get(".xl-react-widget-modal-dialog .title:contains('Override notification settings')");
    }

    findInheritConfirmationModalOkElement() {
        return this.getInheritConfirmationModalElement().closest('.xl-react-modal-content').find("button:contains('Ok')");
    }

    findInheritConfirmationModalCancelElement() {
        return this.getInheritConfirmationModalElement().closest('.xl-react-modal-content').find("button:contains('Cancel')");
    }

    expectInheritMessagePresent(linkText) {
        this.getInheritParagraph().contains(`Inherit notification settings from`).should('be.visible');
        this.getInheritAnchorTag().contains(`${linkText}`);
        return this;
    }

    expectInheritCheckboxToBeChecked(shouldBeChecked = true) {
        const checkedExpectancy = shouldBeChecked ? 'to.be.checked' : 'to.not.be.checked';
        this.getInheritCheckbox().scrollIntoView().should('be.visible').and(checkedExpectancy);
        return this;
    }

    expectCorrectInheritFolderUrl(folderId) {
        this.getInheritAnchorTag().should('have.attr', 'href', `#folders/${folderId}/notifications`);
        return this;
    }

    expectInheritConfirmationModalToBePresent(shouldBePresent = true) {
        const existExpectancy = shouldBePresent ? 'exist' : 'not.exist';
        this.getInheritConfirmationModalElement().should(existExpectancy);
        return this;
    }

    expectResetAndSaveButtonsToBeEnabled(shouldBeEnabled = true) {
        const enabledExpectancy = shouldBeEnabled ? 'to.be.enabled' : 'to.not.be.enabled';
        this.getResetButton().should(enabledExpectancy);
        this.getSaveButton().should(enabledExpectancy);
        return this;
    }

    expectLastSaveTimeToBePresent(shouldBePresent = true) {
        const existExpectancy = shouldBePresent ? 'exist' : 'not.exist';
        this.getLastSavedTimeElement().should(existExpectancy);
        return this;
    }

    uncheckInheritCheckbox() {
        this.getInheritCheckbox().uncheck();
        return this;
    }

    checkInheritCheckbox() {
        this.getInheritCheckbox().check();
        return this;
    }

    resetChanges() {
        this.getResetButton().click();
        return this;
    }

    saveChanges() {
        this.getSaveButton().click();
        return this;
    }

    confirmFromInheritConfirmationModal() {
        this.findInheritConfirmationModalOkElement().click();
        return this;
    }

    cancelFromInheritConfirmationModal() {
        this.findInheritConfirmationModalCancelElement().click();
        return this;
    }

    checkInheritWhenOwnConfigPresentWithAssertions() {
        this.expectInheritCheckboxToBeChecked(false)
            .checkInheritCheckbox()
            .expectInheritConfirmationModalToBePresent();
        return this;
    }

    checkInheritAndRejectOwnConfigWithAssertions() {
        this.checkInheritWhenOwnConfigPresentWithAssertions()
            .confirmFromInheritConfirmationModal()
            .expectInheritCheckboxToBeChecked()
            .expectTablesToBeInReadOnlyMode()
            .expectInheritMessagePresent('Global notification settings')
            .expectResetAndSaveButtonsToBeEnabled(false);
        return this;
    }

    checkInheritWithOwnConfigButCancelFromDialog() {
        this.checkInheritWhenOwnConfigPresentWithAssertions()
            .cancelFromInheritConfirmationModal()
            .expectInheritCheckboxToBeChecked(false)
            .expectTablesToBeInReadOnlyMode(false);
        return this;
    }

    assertInheritanceFromGlobalSettings() {
        this.expectInheritMessagePresent('Global notification settings')
            .expectInheritCheckboxToBeChecked()
            .expectTablesToBeInReadOnlyMode()
            .expectResetAndSaveButtonsToBeEnabled(false);
        return this;
    }

    assertInheritanceFromFolder(folderTitle) {
        this.expectInheritMessagePresent(folderTitle)
            .expectInheritCheckboxToBeChecked()
            .expectTablesToBeInReadOnlyMode()
            .expectResetAndSaveButtonsToBeEnabled(false);
    }

    disinheritAndAssert() {
        this.expectInheritCheckboxToBeChecked()
            .uncheckInheritCheckbox()
            .expectInheritCheckboxToBeChecked(false)
            .expectTablesToBeInReadOnlyMode(false)
            .expectResetAndSaveButtonsToBeEnabled()
            .saveChanges()
            .expectInheritCheckboxToBeChecked(false)
            .expectTablesToBeInReadOnlyMode(false)
            .expectResetAndSaveButtonsToBeEnabled(false);
        return this;
    }

    disinheritButResetChangesAndAssert() {
        this.expectInheritCheckboxToBeChecked()
            .uncheckInheritCheckbox()
            .expectInheritCheckboxToBeChecked(false)
            .expectTablesToBeInReadOnlyMode(false)
            .expectResetAndSaveButtonsToBeEnabled()
            .resetChanges()
            .expectResetAndSaveButtonsToBeEnabled(false)
            .expectTablesToBeInReadOnlyMode();
        return this;
    }

    disinheritButInheritAgainWithoutSaveAndAssert() {
        this.expectInheritCheckboxToBeChecked()
            .uncheckInheritCheckbox()
            .expectInheritCheckboxToBeChecked(false)
            .expectTablesToBeInReadOnlyMode(false)
            .expectResetAndSaveButtonsToBeEnabled()
            .checkInheritCheckbox()
            .expectInheritCheckboxToBeChecked(true)
            .expectTablesToBeInReadOnlyMode(true)
            .expectResetAndSaveButtonsToBeEnabled(false);
    }

    disinheritChangeMessageSaveFromModalAndAssert(eventName, {priority, subject, body}) {
        this.expectInheritCheckboxToBeChecked()
            .uncheckInheritCheckbox()
            .expectInheritCheckboxToBeChecked(false)
            .expectTablesToBeInReadOnlyMode(false)
            .changeEmailDataSaveChangesAndAssert(eventName, {priority, subject, body})
            .expectInheritCheckboxToBeChecked(false)
            .expectResetAndSaveButtonsToBeEnabled(false)
            .openModalAndExpectData(eventName, {priority, subject, body});
    }

    removeRoleButResetChangesAndAssert(eventName, roleName) {
        this.expectTablesToBeInReadOnlyMode(false)
            .removeRoleTagFromEvent(eventName, roleName)
            .expectRoleTagToBePresentOnEvent(eventName, roleName, false)
            .expectResetAndSaveButtonsToBeEnabled()
            .resetChanges()
            .expectRoleTagToBePresentOnEvent(eventName, roleName)
            .expectResetAndSaveButtonsToBeEnabled(false);
        return this;
    }

    removeRoleSaveChangesAndAssert(eventName, roleName) {
        this.expectTablesToBeInReadOnlyMode(false)
            .removeRoleTagFromEvent(eventName, roleName)
            .expectRoleTagToBePresentOnEvent(eventName, roleName, false)
            .expectResetAndSaveButtonsToBeEnabled()
            .saveChanges()
            .expectRoleTagToBePresentOnEvent(eventName, roleName, false)
            .expectResetAndSaveButtonsToBeEnabled(false);
        return this;
    }

    addRoleToEventWithAssertions(eventName, roleName) {
        this.typeRoleTagNameInEventInput(eventName, roleName)
            .expectSuggestedRoleToBeVisible(roleName)
            .selectSuggestedRole(roleName)
            .expectRoleTagToBePresentOnEvent(eventName, roleName)
            .expectResetAndSaveButtonsToBeEnabled();
        return this;
    }

    addRoleToEventButResetChangesAndAssert(eventName, roleName) {
        this.addRoleToEventWithAssertions(eventName, roleName)
            .resetChanges()
            .expectRoleTagToBePresentOnEvent(eventName, roleName, false)
            .expectResetAndSaveButtonsToBeEnabled(false);
        return this;
    }

    addRoleToEventSaveChangesAndAssert(eventName, roleName) {
        this.addRoleToEventWithAssertions(eventName, roleName)
            .saveChanges()
            .expectRoleTagToBePresentOnEvent(eventName, roleName, true)
            .expectResetAndSaveButtonsToBeEnabled(false);
        return this;
    }

    assertRolesAdditionWithSave(eventName, roles) {
        roles.forEach(role => this.addRoleToEventSaveChangesAndAssert(eventName, role));
    }
}
