import Modal from '../components/modal';
import DateTimePicker from "../components/date-time-picker";
import {XlrListStringWithVariables} from "../components/xlr-input-with-variables";
import {zip} from 'lodash';

export default class GlobalVariablesPage {
    constructor() {
        cy.get('.sub-nav-item.active').should('contain', "Global variables");
        return this;
    }

    openNewVariableModal() {
        // refresh page until ".button.primary" will appear
        cy.doWithRetry(
            () => {},
            '.button.primary',
            () => cy.reload()
                .log('this step is just for videos, to check whether')
                .log('button will appear after page reload')
                .then(() => cy.wait(20000))
                .then(() => {throw new Error();}),
        );
        cy.get('.button.primary').click();
        return new GlobalVariableDetails();
    }

    openDeleteModal(variable) {
        cy.get(`.variable:contains('${variable}') .action.delete`).click();
        return this;
    }

    delete() {
        cy.get("#modal button.delete").click();
        return this;
    }

    setGlobalVariableSearchFilter(filterText) {
        cy.get('input#variables-filter').clear();
        if (filterText) {
            cy.get('input#variables-filter').type(filterText);
        }
        return this;
    }

    switchToggle() {
        cy.get('.variable-toggle button').click();
        return this;
    }

    edit(name) {
        cy.get(`.variable:contains('${name}') .action.edit`).click();
        return new GlobalVariableDetails();
    }

    selectFor(currentVar, newVar) {
        cy.get(`.variable-editor #${currentVar} .ui-autocomplete-input`).focus();
        cy.get(`.ui-autocomplete li.ui-menu-item:contains(${newVar})`).click();
        return this;
    }

    setValue(newValue) {
        cy.get('.editor input.editable').clear().type(newValue);
        return this;
    }

    replaceAndDelete() {
        cy.get('button.delete').click();
    }

    expectVariablesDisplayed(variables) {
        for (const name in variables) {
            const expected = variables[name];
            const expectation = expected ? 'be.visible' : 'not.exist';
            cy.get(`.variable:contains('${variables[name]}')`).scrollIntoView();
            cy.get(`.variable:contains('${variables[name]}')`).should(expectation);
        }
        return this;
    }

    expectContainingVariable(name, value) {
        cy.get(`.variable:contains('${name}')`).should('have.length', 1);
        cy.get(`.variable:contains('${name}'):contains('${value}')`).should('have.length', 1);
        return this;
    }

    expectNoVariables() {
        cy.get('.variables-list').contains('No variables defined.');
        return 0;
    }

    expectHeader(label) {
        cy.get('.variables-list .header-row').contains(`${label}`);
        return this;
    }

    expectVariablesCountToBe(expectedCount) {
        cy.get(".variable").should('have.length', expectedCount);
        return this;
    }

    expectVariablesCountToBeAtLeast(expectedCount) {
        cy.get(".variable").its('length').should('be.gte', expectedCount);
        return this;
    }

    expectGlobalVariableDisplayed(variable) {
        const varElement = cy.get(`.variable:contains('${variable}')`);
        varElement.scrollIntoView();
        varElement.should('be.visible');
        return this;
    }

    expectVariableWithNameValueAndType(name, value, type) {
        cy.get(`.variable:contains('${name}'):contains('${value}'):contains('${type}')`).should('be.visible');
        return this;
    }

    expectVariableWithLabel(label) {
        cy.get(`.variable:contains('${label}')`);
        return this;
    }

    expectDateVariableWithNameAndValue(name, value, format) {
        cy.get(`.variable:contains('${name}') .data-row`).click();
        cy.compareToLocalDate(value, '#modal .date input', format);
        return this;
    }

    expectVariablesDisplayedInOrder(variableNames) {
        cy.get(".variable > div > div:first-child > div").then((elements) => {
            expect(elements.length).to.eq(variableNames.length);
            for (const [name, item] of zip(variableNames, elements)) {
                if (item) {
                    expect(item).to.have.text(name);
                }
            }
        });
        return this;
    }

    drag(from, to) {
        cy.get(`.variable:contains('${from}')`).trigger("mousedown", {which: 1});
        cy.get(`.variable:contains('${to}')`)
            .trigger("mousemove")
            .trigger("mouseup");
        return this;
    }

    expectReplacementPromptDisplayed(visible) {
        cy.get("div.variable-replacement").should((visible) ? 'be.visible' : 'not.exist');
        return this;
    }

    expectVariableNotDisplayed(variableName) {
        cy.get(`.variable:contains('${variableName}')`).should('not.exist');
        return this;
    }

    expectNewVariableButtonDisplayed(expected = true) {
        cy.get(".button.new-variable").should((expected) ? 'be.visible' : 'not.exist');
        return this;
    }

    expectEditVariableButtonsEnabled(expected = true) {
        cy.get(".variable .action.edit:enabled").should((expected) ? 'be.visible' : 'not.exist');
        return this;
    }

    expectDeleteVariableButtonsEnabled(expected = true) {
        cy.get(".variable .action.delete:enabled").should((expected) ? 'be.visible' : 'not.exist');
        return this;
    }

}

class GlobalVariableDetails extends Modal {

    setName(name) {
        cy.get('.input-group > .input-block-level').clear();
        if (name !== '') {
            cy.get('.input-group > .input-block-level').type(name);
        }
        return this;
    }

    selectType(type) {
        cy.get(`select.input-block-level`).select(type);
        return this;
    }

    setTextValue(value) {
        cy.get('.variable-value').click();
        cy.get('.variable-value input.input-block-level').type(value);
        return this;
    }

    setPasswordValue(value) {
        cy.get('#modal input[data-test-id="password-input"]').clear().type(value);
        return this;
    }

    togglePasswordField() {
        cy.wait(500);
        cy.get('#modal .btn-toggle-lookup').click();
        return this;
    }

    setPasswordValueFromConfigurationServer(serverName, path, key) {
        cy.doWithRetry(
            () => cy.get('#modal xl-react-widget-autocomplete[data-test-id="external-server"] input').type(serverName)
                .get(`#modal xl-react-widget-autocomplete[data-test-id="external-server"] .xl-react-autocomplete-suggestion-list-container .xl-react-autocomplete-suggestion:contains('${serverName}')`).click({force: true}),
            `#modal xl-react-widget-autocomplete[data-test-id="external-server"] input[value="${serverName}"]`
        );
        cy.get('#modal input[data-test-id="external-path"]').clear().type(path);
        cy.get('#modal input[data-test-id="external-key"]').clear().type(key);
        return this;
    }

    setMulitineTextValue(value) {
        cy.get('#modal textarea[ng-model="var.value"]').clear();
        cy.get('#modal textarea[ng-model="var.value"]').type(value);
        return this;
    }

    clearTextValue() {
        cy.get('#modal .variable-value input"]').clear();
        return this;
    }

    clickMultiline() {
        cy.get('#modal #multiline').click();
        return this;
    }

    addPossibleValue(value) {
        this._possibleValues()
            .editor()
            .enter(value)
            .add();
        return this;
    }

    choosePossibleValuesVariable(name) {
        this._possibleValues()
            .toggleMode()
            .variableSelector()
            .focus()
            .select(name);
        return this;
    }

    createPossibleValuesVariable(name) {
        this._possibleValues()
            .toggleMode()
            .variableSelector()
            .focus()
            .enter(name)
            .select(name);
        return this;
    }

    expectPossibleValuesVariable(name) {
        this._possibleValues()
            .expectVariableMode()
            .variableSelector()
            .expectValue(name);
        return this;
    }

    selectValue(value) {
        cy.get(`#modal select:contains('${value}')`).select(value);
        return this;
    }

    clickCheckbox() {
        cy.get('#modal input[ng-model="var.value"]').click();
        return this;
    }

    setLabel(value) {
        cy.get('#modal .variable-label input').clear();
        cy.get('#modal .variable-label input').type(value);
        return this;
    }

    setDescription(value) {
        cy.get('#modal .variable-description input').clear();
        cy.get('#modal .variable-description input').type(value);
        return this;
    }

    save() {
        cy.get('#modal button.save').click();
        cy.get('#modal').should('not.be.visible');
        return new GlobalVariablesPage();
    }

    cancel() {
        cy.get("button.cancel").click();
        return this;
    }

    close() {
        return cy.get('.close').click();
    }

    expectSaveButtonInvisible() {
        cy.get("#modal button.save:visible").should('have.length', 0);
        return this;
    }

    expectSaveButtonDisabled() {
        cy.get(".save:disabled").should('have.length', 1);
        return this;
    }

    expectSaveButtonEnabled() {
        cy.get("#modal button.save:enabled").should('have.length', 1);
        return this;
    }

    expectLabel(label) {
        cy.get(`#modal .control-label:contains('${label}')`).should('be.visible');
        return this;
    }

    expectValueToBe(value) {
        cy.get(".variable-value select option[selected='selected']").should('contain', value);
        return this;
    }

    expectPasswordValueToBe(value) {
        cy.get('#modal input[data-test-id="password-input"]').should('have.value', value);
        return this;
    }
    expectTextareaValueToBe(value) {
        cy.get('#modal textarea[ng-model="var.value"]').should('have.value', value);
        return this;
    }

    expectNotToBeEditable(value) {
        cy.get(`#modal .variable-name span:contains('${value}')`).should('be.visible');
    }

    expectName(value) {
        cy.get(`#modal .variable-name:contains('${value}')`).should('be.visible');
        return this;
    }

    expectNameInput(value) {
        cy.get(`#modal .variable-label input[ng-model="var.key"]`).should('have.value', value);
        return this;
    }

    expectLabelAndDescription(label, desc) {
        cy.get(`#modal .variable-label input[ng-model="var.label"]`).should('have.value', label);
        cy.get(`#modal .variable-description input[ng-model="var.description"]`).should('have.value', desc);
        return this;
    }

    expectRequired(value) {
        cy.get(`#modal .variable-label input[ng-model="var.requiresValue"]`).should('have.value', value ? 'on' : 'off');
        return this;
    }

    expectRequiredFieldNotPresent() {
        cy.get(`#modal .variable-label input[ng-model="var.requiresValue"]`).should('not.exist');
        return this;
    }

    expectShowOnReleaseStart(value) {
        cy.get(`#modal .variable-label input[ng-model="var.showOnReleaseStart"]`).should('have.value', value ? 'on' : 'off');
        return this;
    }

    expectShowOnReleaseStartFieldNotPresent() {
        cy.get(`#modal .variable-label input[ng-model="var.showOnReleaseStart"]`).should('not.exist');
        return this;
    }

    expectNoValidationErrors() {
        cy.get("#modal .field-errors span:visible").should('not.exist');
        return this;
    }

    expectValidationError(text) {
        cy.get(`#modal .field-errors span:visible:contains('${text}')`).should('be.visible');
        return this;
    }

    expectPossibleValues(values) {
        this._possibleValues()
            .editor()
            .shouldHaveItems(values);
        return this;
    }

    expectPasswordFromServerVariableValue(serverName, path, key) {
        cy.get('#modal xl-react-widget-autocomplete[data-test-id="external-server"] input').should('have.value', serverName);
        cy.get('#modal input[data-test-id="external-path"]').should('have.value', path);
        cy.get('#modal input[data-test-id="external-key"]').should('have.value', key);
        return this;
    }

    expectPasswordFromServerVariableLabel(serverName, path, key) {
        cy.get('#modal p[data-test-id="readonly-value"]').should('contain', serverName);
        cy.get('#modal p[data-test-id="external-path"]').should('contain', path);
        cy.get('#modal p[data-test-id="external-key"]').should('contain', key);
        return this;
    }

    expectPasswordLabel() {
        cy.get('#modal p[data-test-id="readonly-value"]').should('contain', '******');
        return this;
    }

    setDateTime(date) {
        const dateTimePicker = new DateTimePicker(`#modal .date-editor:first`);
        dateTimePicker.setDate({day: date.day, month: date.month, year: date.year});
        dateTimePicker.setTime({hour: date.hour, minute: date.minute}, () => cy.get('.modal-dialog label.control-label:first').click());
        return this;
    }

    _possibleValues() {
        return new XlrListStringWithVariables('#modal div[name=expanded-value-provider-value]');
    }
}
