import Modal from '../components/modal';
import _ from "lodash";
import DateTimePicker from "../components/date-time-picker";

export default class ReleaseVariablesPage {
    constructor() {
        cy.get('.variables-list').should('be.visible');
    }

    openDeleteModal(variable) {
        cy.get(`.variable:contains('${variable}') .action.delete`).click();
        return this;
    }

    switchToggle() {
        cy.get('.variable-toggle button').click();
        return this;
    }

    openNewVariableModal() {
        cy.get('.button.primary').click();
        return new VariableDetails();
    }

    edit(name) {
        cy.get(`.variable:contains('${name}') .action.edit`).click();
        return new VariableDetails();
    }

    setSearchFilter(filterText) {
        cy.get('input#variables-filter').clear();
        if (filterText) {
            cy.get('input#variables-filter').type(filterText);
        }
        return this;
    }

    selectFor(currentVar, newVar) {
        cy.get(`.variable-editor #${currentVar} .ui-autocomplete-input`).focus();
        cy.get(`.ui-autocomplete li.ui-menu-item:contains(${newVar})`).click();
        return this;
    }

    setValue(newValue) {
        cy.get('.editor input.editable').clear().type(newValue);
        return this;
    }

    replaceAndDelete() {
        cy.get('button.delete').click();
    }

    drag(from, to, shouldUpdateRelease) {
        if (shouldUpdateRelease) {
            cy.server();
            cy.route('PUT', '**/api/v1/releases/**').as('updateVariable');
        }
        cy.get(`.variable:contains('${from}')`).trigger("mousedown", {which: 1});
        cy.get(`.variable:contains('${to}')`)
            .trigger("mousemove")
            .trigger("mouseup");
        if (shouldUpdateRelease) {
            cy.wait('@updateVariable');
        }
        return this;
    }

    showFolderVariables() {
        cy.get('a:contains("Folder variables")').click();
        return this;
    }

    expectContainingVariable(name, value) {
        cy.get(`.variable:contains('${name}')`).should('have.length', 1);
        cy.get(`.variable:contains('${name}'):contains('${value}')`).should('have.length', 1);
        return this;
    }

    expectVariableWithNameValueAndType(name, value, type) {
        cy.get(`.variable:contains('${name}'):contains('${value}'):contains('${type}')`).should('be.visible');
        return this;
    }

    expectCountOfVariables(count) {
        cy.get('.variable').find('.data-row').should('have.length', count);
        return this;
    }

    expectNoVariables() {
        cy.get('.variables-list').contains('No variables defined.');
        return this;
    }

    expectHeader(label) {
        cy.get('.variables-list .header-row').should('contain', `${label}`);
        return this;
    }

    expectVariablesDisplayed(variables) {
        Object.entries(variables).forEach(([name, value]) => {
            if (value) {
                cy.get(`.variable:contains('${name}')`).scrollIntoView();
            }
            const expectation = value ? 'be.visible' : 'not.exist';
            cy.get(`.variable:contains('${name}')`).should(expectation);
        });
        return this;
    }

    expectActionsDisplayed(actions) {
        Object.entries(actions).forEach(([name, value]) => {
            const expectation = value ? 'be.visible' : 'not.exist';
            cy.get(`.action:contains('${name}')`).should(expectation);
        });
        return this;
    }


    expectVariablesDisplayedInOrder(variableNames) {
        cy.get(".variable > div > div:first-child > div").then(elements => {
            expect(elements.length).to.eq(variableNames.length);
            for (const [name, item] of _.zip(variableNames, elements)) {
                if (item) {
                    expect(item).to.have.text(name);
                }
            }
        });
        return this;
    }

    expectEditVariableButtonsEnabled(expected = true) {
        cy.get(".variable .action.edit:enabled").should(expected ? 'be.visible' : 'not.exist');
        return this;
    }
}

class VariableDetails extends Modal {
    // constructor() {
    // super(...arguments);
    // this.variableDetails = element(By.id('modal'));
    // }

    setName(name) {
        if (name) {
            cy.get('.variable-name input').clear().type(name);
        } else {
            cy.get('.variable-name input').clear();
        }
        return this;
    }

    selectType(type) {
        cy.get(`#modal select.input-block-level`).select(type);
        return this;
    }

    setTextValue(value) {
        cy.get('#modal input[ng-model="var.value"]').clear();
        cy.get('#modal input[ng-model="var.value"]').type(value);
        return this;
    }

    clearTextValue() {
        cy.get('#modal input[ng-model="var.value"]').clear();
        return this;
    }

    setPasswordValue(value) {
        cy.get('#modal input[data-test-id="password-input"]').clear().type(value);
        return this;
    }

    togglePasswordField() {
        cy.wait(500);
        cy.get('#modal .btn-toggle-lookup').click();
        return this;
    }

    setPasswordValueFromConfigurationServer(serverName, path, key) {
        cy.doWithRetry(
            () => cy.get('#modal xl-react-widget-autocomplete[data-test-id="external-server"] input').type(serverName)
                .get(`#modal xl-react-widget-autocomplete[data-test-id="external-server"] .xl-react-autocomplete-suggestion-list-container .xl-react-autocomplete-suggestion:contains('${serverName}')`).click({force: true}),
            `#modal xl-react-widget-autocomplete[data-test-id="external-server"] input[value="${serverName}"]`
        );
        cy.get('#modal input[data-test-id="external-path"]').clear().type(path);
        cy.get('#modal input[data-test-id="external-key"]').clear().type(key);
        return this;
    }

    clickMultiline() {
        cy.get('#modal #multiline').click();
        return this;
    }

    setMulitineTextValue(value) {
        cy.get('#modal textarea[ng-model="var.value"]').clear();
        cy.get('#modal textarea[ng-model="var.value"]').type(value);
        return this;
    }

    setPasswordMultilineValue(value) {
        cy.get('#password-lookup-field .password-textarea').clear();
        cy.get('#password-lookup-field .password-textarea').type(value);
        return this;
    }

    addPossibleValue(value) {
        this._possibleValues()
            .editor()
            .enter(value)
            .add();
        return this;
    }

    choosePossibleValuesVariable(name) {
        this._possibleValues()
            .toggleMode()
            .variableSelector()
            .focus()
            .select(name);
        return this;
    }

    createPossibleValuesVariable(name) {
        this._possibleValues()
            .toggleMode()
            .variableSelector()
            .focus()
            .enter(name)
            .select(name);
        return this;
    }

    expectPossibleValuesVariable(name) {
        this._possibleValues()
            .expectVariableMode()
            .variableSelector()
            .expectValue(name);
        return this;
    }

    selectValue(value) {
        cy.get(`#modal option:contains('${value}')`).click();
        return this;
    }

    clickCheckbox() {
        cy.get('#modal input[ng-model="var.value"]').click();
        return this;
    }

    setLabel(value) {
        cy.get('#modal input[ng-model="var.label"]').clear().type(value);
        return this;
    }

    setDescription(value) {
        cy.get('#modal input[ng-model="var.description"]').clear().type(value);
        return this;
    }

    save() {
        cy.get('#modal button.save').click();
        cy.get('#modal').should('not.be.visible');
        return this;
    }

    cancel() {
        cy.get('#modal .button.cancel').click();
        return this;
    }

    close() {
        return cy.get('.close').click();
    }

    expectSaveButtonInvisible() {
        cy.get('#modal button.save:visible').should('have.length', 0);
        return this;
    }

    expectSaveButtonDisabled() {
        cy.get('#modal button.save:disabled').should('have.length', 1);
        return this;
    }

    expectSaveButtonEnabled() {
        cy.get('#modal button.save:enabled').should('have.length', 1);
        return this;
    }

    expectLabel(label) {
        cy.get(`#modal .control-label:contains('${label}')`).should('be.visible');
        return this;
    }

    expectValueToBe(value) {
        cy.get('#modal .variable-value div').then(element => {
            if (element.children().is('[ng-model]')) {
                cy.get('#modal input[ng-model="var.value"]').should('have.value', value);
            } else {
                expect(element.text()).to.have.string(value);
            }
        });
        return this;
    }

    expectTextareaValueToBe(value) {
        cy.get('#modal textarea[ng-model="var.value"]').should('have.value', value);
        return this;
    }

    expectNotToBeEditable(value) {
        cy.get(`#modal .variable-name span:contains('${value}')`).should('be.visible');
        return this;
    }

    expectName(value) {
        cy.get(`#modal .variable-name:contains('${value}')`).should('be.visible');
        return this;
    }

    expectNameInput(value) {
        cy.get('#modal input[ng-model="var.key"]').should('have.value', value);
        return this;
    }

    expectLabelAndDescription(label, desc) {
        cy.get('#modal input[ng-model="var.label"]').should('have.value', label);
        cy.get('#modal input[ng-model="var.description"]').should('have.value', desc);
        return this;
    }

    expectRequired(value) {
        cy.get('#modal input[ng-model="var.requiresValue"]').should('have.value', value ? 'on' : 'off');
        return this;
    }

    expectRequiredFieldNotPresent() {
        cy.get('#modal input[ng-model="var.requiresValue"]').should('not.exist');
        return this;
    }

    expectShowOnReleaseStart(value) {
        cy.get('#modal input[ng-model="var.showOnReleaseStart"]').should('have.value', value ? 'on' : 'off');
        return this;
    }

    expectShowOnReleaseStartFieldNotPresent() {
        cy.get('#modal input[ng-model="var.showOnReleaseStart"]').should('not.exist');
        return this;
    }


    expectRequiredChecked(expected = true) {
        cy.get(`#modal input[ng-model="var.requiresValue"]`).should(`${expected ? '' : 'not.'}be.checked`);
        return this;
    }

    expectDisabledRequired(expected = true) {
        cy.get(`#modal input[ng-model="var.requiresValue"]`).should(`${expected ? '' : 'not.'}be.disabled`);
        return this;
    }

    expectShowOnReleaseStartChecked(expected = true) {
        cy.get(`#modal input[ng-model="var.showOnReleaseStart"]`).should(`${expected ? '' : 'not.'}be.checked`);
        return this;
    }

    expectDisabledShowOnReleaseStart(expected = true) {
        cy.get(`#modal input[ng-model="var.showOnReleaseStart"]`).should(`${expected ? '' : 'not.'}be.disabled`);
        return this;
    }

    expectNoValidationErrors() {
        cy.get('#modal .field-errors span:visible').should('not.exist');
        return this;
    }

    expectValidationError(text) {
        cy.get(`#modal .field-errors span:visible:contains('${text}')`).should('be.visible');
        return this;
    }

    expectPossibleValues(values) {
        this._possibleValues()
            .editor()
            .shouldHaveItems(values);
        return this;
    }

    setDateTime(date) {
        const dateTimePicker = new DateTimePicker(`#modal .date-editor:first`);
        dateTimePicker.setDate({day: date.day, month: date.month, year: date.year});
        dateTimePicker.setTime({hour: date.hour, minute: date.minute}, () => cy.get('.modal-dialog label.control-label:first').click());
        return this;
    }

    expectDateVariableValue(dateTime) {
        const dateTimeFormat = {
            year: "2-digit",
            month: "numeric",
            day: "numeric",
            hour: "numeric",
            minute: "numeric"
        };
        const dateTimePicker = new DateTimePicker(`#modal .date-editor`);
        dateTimePicker.expectDateTimeToBe(dateTime, dateTimeFormat);

        return this;
    }

    expectPasswordFromServerVariableValue(serverName, path, key) {
        cy.get('#modal xl-react-widget-autocomplete[data-test-id="external-server"] input').should('have.value', serverName);
        cy.get('#modal input[data-test-id="external-path"]').should('have.value', path);
        cy.get('#modal input[data-test-id="external-key"]').should('have.value', key);
        return this;
    }

    _possibleValues() {
        return new XlrListStringWithVariables('#modal div[name=expanded-value-provider-value]');
    }
}
