import ReleaseVariablesPage from './release-variables-page';
import ColorPicker from '../components/color-picker';
import ReleaseSummaryPage from './release-summary-page';
import ReleaseGanttPage from './release-gantt-page';
import {ReleasePropertiesPage} from './release-properties-page';
import {
    CreateReleaseTaskDetails,
    CustomScriptDetails,
    ExternalScriptTaskDetails,
    GateDetails,
    NotificationDetails,
    ParallelGroupDetails,
    TaskDetails,
    UserInputDetails
} from '../components/task-details';
import {InlineEditor} from './inline-editor';
import ReleasePermissionsPage from './release-permissions';
import ReleaseGridPage from './release-grid-page';
import Page from './page';
import {ScriptDetails} from "../components/task-details/script-task-details";
import TagEditor from '../components/tag-editor';
import RestartPhaseModal from '../components/restart-phase-modal';
import {TemplatePropertiesPage} from './properties-page';
import TriggersPage from "./triggers-page";

export default class ReleasePage {
    constructor(releaseId) {
        this.releaseId = releaseId;
        cy.get('#release-content').should('be.visible');
    }

    open(name) {
        cy.waitForAngular();
        return cy
            .doWithRetry(
                () => cy
                    .get('.subpage-links .dropdown-button')
                    .click(),
                () => expect(Cypress.$(`#subpages-container div:contains('${name}')`)).to.exist)
            .get(`#subpages-container div:contains('${name}')`)
            .click();
    }

    expectTitleToBe(title) {
        cy.get('.bread-crumbs span').should('contain', title);
        return this;
    }

    openPermissions() {
        this.openSubPage('Permissions');
        return new ReleasePermissionsPage();
    }

    newRelease() {
        cy.get('.new-release').click();
        return new ReleasePropertiesPage();
    }

    openTemplateProperties() {
        this.openSubPage('Properties');
        return new TemplatePropertiesPage();
    }

    expectTaskCompletedInAdvance(title) {
        cy.get(`.task:contains('${title}').completed_in_advance`, {timeout : 2000}).should('have.length', 1);
        return this;
    }

    openScriptDetails(title) {
        return this.openTaskDetails(title, () => new ScriptDetails());
    }

    exportToZip() {
        cy.server();
        cy.route({
            method: 'GET',
            url: '**/api/v1/templates/zip/*',
            status: 304, //prevents the browser from opening the file download dialog, so the test won't hang
            response: []
        }).as('getZip');
        this.exportTemplateAs('XLR');
        cy.wait('@getZip');
    }

    downloadAuditReportExcel() {
        cy.server();
        cy.route({
            method: 'GET',
            url: '**/api/v1/reports/download/*',
            status: 304, //prevents the browser from opening the file download dialog, so the test won't hang
            response: []
        }).as('getExcel');
        this.exportTemplateAs('Release audit report');
        cy.wait('@getExcel');
        return this;
    }

    waitForTaskPlanned(title) {
        cy.get(`.task:contains('${title}').planned`, {timeout : 2000}).should('be.visible');
        return this;
    }

    waitForTaskSkippedInAdvance(title) {
        cy.get(`.task:contains('${title}').skipped_in_advance`, {timeout : 2000}).should('be.visible');
        return this;
    }

    navigateToPhase(phaseTitle) {
        cy.get('#navigator:visible').should('have.length', 0);
        cy.get('#navigator-toggler').click();
        cy.get('#navigator:visible').should('have.length', 1);
        cy.get(`#navigator .phase-link:contains('${phaseTitle}')`).click();
        cy.get('#navigator-toggler').click();
        cy.get('#navigator:visible').should('have.length', 0);
        return this;
    }

    exportTemplateAs(name) {
        cy.get('.export-buttons.dropdown-button').click();
        cy.get(`.export-buttons-container div:contains('${name}')`).click();
        return this;
    }

    openGantt() {
        this.selectView('Planner');
        return new ReleaseGanttPage();
    }

    openParallelGroupDetails(title) {
        return this.openTaskDetails(title, () => new ParallelGroupDetails());
    }

    expectSubPageTitleToBe(title) {
        cy.get('#release-header .subpage-links button').should('contain', title);
        return this;
    }

    addPhase() {
        cy.get('.add-phase.button:visible').click();
        return this;
    }

    openNotificationDetails(title) {
        return this.openTaskDetails(title, () => new NotificationDetails());
    }

    openManualTaskDetails(title) {
        return this.openTaskDetails(title, () => new TaskDetails());
    }

    openCustomScriptDetails(title) {
        return this.openTaskDetails(title, () => new CustomScriptDetails());
    }

    openCreateReleaseTaskDetails(title) {
        return this.openTaskDetails(title, () => new CreateReleaseTaskDetails());
    }

    openGateTaskDetails(title) {
        return this.openTaskDetails(title, () => new GateDetails());
    }

    openUserInputDetails(title) {
        return this.openTaskDetails(title, () => new UserInputDetails());
    }

    openExternalScriptTaskDetails(title) {
        return this.openTaskDetails(title, () => new ExternalScriptTaskDetails());
    }

    openScriptTaskDetails(title) {
        return this.openTaskDetails(title, () => new ScriptDetails());
    }

    openUserInputTaskDetails(title) {
        return this.openTaskDetails(title, () => new UserInputDetails());
    }

    expectTaskCompleted(title) {
        cy.get(`.task:contains('${title}').completed`).should('have.length',1);
        return this;
    }

    expectCurrentTasksToContain(title) {
        cy.get('.task.active .task-title').should('contain',title);
        return this;
    }

    waitForTaskCompletedInAdvance(title) {
        cy.get(`.task:contains('${title}').completed_in_advance`).should('be.visible');
        return this;
    }

    openVariables() {
        this.open('Variables');
        return new ReleaseVariablesPage();
    }

    openFlow() {
        return this.open('Release flow');
    }

    openGateDetails(title) {
        return this.openTaskDetails(title, () => new GateDetails());
    }

    openEditor() {
        this.openSubPage('Release flow');
        return new ReleasePage();
    }

    openGrid() {
        this.selectView('Table');
        return new ReleaseGridPage();
    }

    selectView(name) {
        cy.get(`#release-header a:contains('${name}')`).click();
        // return element(By.$(`#release-header a:contains('${name}')`)).click();
    }

    openTeams() {
        this.openSubPage('Permissions');
        return new Teams();
    }

    openTaskDetails(title, buildResult) {
        cy.server();
        cy.route('GET', /tasks\/.*\/watchers/).as('openTaskDetails1');
        cy.route('GET', /tasks\/.*\/comments/).as('openTaskDetails2');
        cy.get(`.task-header:contains('${title}')`).click();
        cy.wait('@openTaskDetails1');
        cy.wait('@openTaskDetails2');
        return buildResult();
    }

    expectTaskDetailsModal(phaseTitle, taskTitle, expected) {
        cy.get(`.modal .modal-dialog .modal-content.task-details`).should(`${expected ? '' : 'not.'}visible`);
        if (expected) {
          cy.get(`.modal .modal-dialog .modal-content.task-details .modal-header .task-title:contains('${taskTitle}')`).should('be.visible');
          cy.get(`.modal .modal-dialog .modal-content.task-details .modal-header .task-info:contains('${phaseTitle}')`).should('be.visible');
        }
    }

    openProperties() {
        this.openSubPage('Properties');
        return new ReleasePropertiesPage();
    }

    getPhase(phaseName) {
        return new ReleasePhase(`.phase:contains('${phaseName}')`);
    }

    expectPhaseCountToBe(expectedPhaseCount) {
        cy.get('.phase').should('have.length', expectedPhaseCount);
        return this;
    }

    expectHasPhase(title) {
        cy.get(`.phase-title:contains('${title}')`).should('be.visible');
        return this;
    }

    expectVariableToBeFound(variableName) {
        cy.get(`span.variable:contains('${variableName}')`).should('have.length', 1);
    }

    start() {
        cy.server();
        cy.route('POST', /releases\/.*\/start/).as('startRelease');
        cy.get('.start-release').click({force: true});
        cy.get(".modal:visible .button:contains('Start')").click();
        cy.wait('@startRelease');
        cy.get('.release-status').should('not.contain', 'Pending');
        return this;
    }

    expectStarted() {
        this.expectStatus('in_progress');
        return this;
    }

    expectAborted() {
        return this.expectStatus('aborted');
    }

    expectCompleted() {
        return this.expectStatus('completed');
    }

    expectStatus(status) {
        cy.get(`#release.${status}`).should('have.length',1);
    }

    openSubPage(name) {
        cy.waitForPopover('.subpage-links .dropdown-button');
        cy.get(`#subpages-container div:contains('${name}')`).click();
        return this;
    }

    expectSubPage(name, toBePresent = true) {
        cy.get('#release-header .subpage-links button').click();
        cy.get(`#subpages-container div:contains('${name}')`).should('be.visible');
        Page.clickElseWhere();
        return this;
    }

    expectTaskToBe(title) {
        cy.get(`.task:contains('${title}')`).should('be.visible');
        return this;
    }

    expectTaskToHaveIcon(title, icon) {
        cy.get(`.task:contains('${title}') .labels ${icon}`).should('be.visible');
        return this;
    }

    expectTaskPending(title) {
        cy.get(`.task:contains('${title}').pending`).should('be.visible');
        return this;
    }

    expectTaskInProgress(title) {
        cy.get(`.task:contains('${title}').in_progress`).should('be.visible');
        return this;
    }

    waitForTaskFailed(title, timeout) {
        cy.get(`.task:contains('${title}').failed`, {timeout}).should('be.visible');
        return this;
    }

    waitForTaskSkipped(title, timeout) {
        cy.get(`.task:contains('${title}').skipped`, {timeout}).should('be.visible');
        return this;
    }

    waitForTaskStarted(title, timeout) {
        cy.get(`.task:contains('${title}').in_progress`, {timeout}).should('be.visible');
        return this;
    }

    waitForTaskCompleted(title) {
        cy.get(`.task:contains('${title}').completed`).should('be.visible');
        return this;
    }

    waitForTaskInProgress(title) {
        cy.get(`.task:contains('${title}').in_progress`).should('be.visible');
        return this;
    }

    waitForTaskInFacetProgress(title) {
        cy.get(`.task:contains('${title}') .label-preconditioninprogress:contains('Verifying')`).should('be.visible');
        return this;
    }

    expectCurrentTasksCountToBe(count) {
        cy.get('.task.active').should('have.length', count);
        return this;
    }

    expectNoCurrentPhase() {
        cy.get('.phase-header.active-phase').should('have.length', 0);
        return this;
    }

    restartPhases() {
        cy.get('#release-header .restart-phases.button').click();
        return new RestartPhaseModal();
    }

    resume() {
        cy.get('#release-header .resume.button').click();
        return this;
    }

    expectCurrentPhaseToBe(phaseName) {
        cy.get(`.phase-header.active-phase:contains('${phaseName}')`).should('have.length', 1);
        return this;
    }

    expectCurrentTaskToContain(title) {
        cy.get(`.task.active .task-title:contains('${title}')`).should('have.length', 1);
        return this;
    }

    expectTaskSkipped(title) {
        cy.get(`.task:contains('${title}') .label:contains('Skipped')`).should('have.length', 1);
        return this;
    }

    viewAllCompletedPhases() {
        cy.get('.phase a:contains("View all completed phases")').click();
        return this;
    }

    waitForTaskNeedsInput(title) {
        cy.get(`.task:contains('${title}').waiting_for_input`).should('be.visible');
        return this;
    }

    openSummary() {
        this.openSubPage('Release dashboard');
        return new ReleaseSummaryPage();
    }

    refresh() {
        Page.softReload();
        return new ReleasePage();
    }

    openTriggers() {
        this.openSubPage('Triggers');
        return new TriggersPage();
    }

    abort(abortComment = "Aborted in testing") {
        cy.get('.abort-release').click();
        cy.get('textarea[ng-model="abortComment"]').type(abortComment);
        cy.get('.modal:visible .continue').click();
    }

    clickOnStatusLine(title) {
        cy.get(`.task .status-line a:contains('${title}')`).click();
        return this;
    }

}

export class PhaseDetails {
    constructor() {

    }

    setTitle(title) {
        const editor = new InlineEditor('.modal:visible .phase-title');
        editor.set(title);
        return this;
    }

    setDescription(description) {
        const editor = new TextareaInlineEditor('.modal:visible .phase-description');
        return editor.set(description);
    }

    getDescription() {
        const editor = new TextareaInlineEditor('.modal:visible .phase-description');
        return editor.value();
    }

    setDueDateDay(day) {
        new DateTimePicker('.modal .due-date .date-editor').setDay(day);
        return this;
    }

    close() {
        cy.get('.modal button.close').click();
        return this;
    }

    expectScheduledStartDateToBeDisplayed() {
        expect(element('.modal .scheduled-start-date').count()).toBe(1);
        return this;
    }

    expectDueDateToBeDisplayed() {
        expect(element('.modal .due-date').count()).toBe(1);
        return this;
    }

    expectStartDateToBeDisplayed() {
        expect(element('.modal .start-date').count()).toBe(1);
        return this;
    }

    expectEndDateToBeDisplayed() {
        expect(element('.modal .end-date').count()).toBe(1);
        return this;
    }

    expectScheduledStartDateDayToBe(day) {
        new DateTimePicker('.scheduled-start-date .date-editor').expectDateToContain(day);
        return this;
    }
}


export class ReleasePhase {
    constructor(phaseSelector) {
        this.phaseSelector = phaseSelector;
        this.phase = cy.get(this.phaseSelector);
    }

    getCurrentTasks() {
        return cy.get(`${this.phaseSelector} .task.active`);
    }

    getFailedTasks() {
        return cy.get(`${this.phaseSelector} .task.failed`);
    }

    getFailingTasks() {
        return cy.get(`${this.phaseSelector} .task.failing`);
    }

    getNumberOfTasks() {
        return cy.get(`${this.phaseSelector} .task`);
    }

    getHeader() {
        return cy.get(`${this.phaseSelector} .phase-header`);
    }

    getColorPicker() {
        return new ColorPicker(`${this.phaseSelector} .pick-color`);
    }

    getTaskGenerator() {
        return new TaskGenerator(this.phaseSelector);
    }

    setTitle(title) {
        const editor = new InlineEditor(`${this.phaseSelector} .phase-title`);
        return editor.set(title);
    }

    getTask(taskName) {
        return new ReleaseTask(`${this.phaseSelector} .task:contains('${taskName}')`);
    }

    expandTaskGroup(taskGroupName) {
        cy.get(`${this.phaseSelector} .task-title:contains(${taskGroupName})`)
            .closest('.task-container')
            .find('.task-box .expander > i')
            .first()
            .then($icon => {
                if ($icon.hasClass('arrow-right-icon')) {
                    $icon.click();
                }
            });
        return this;
    }

    delete() {
        cy.get(`${this.phaseSelector} .delete-phase`).click({force:true});
        cy.get('.modal:visible .modal-header').should('contain', 'Delete Phase');
        return cy.get(".modal:visible .button:contains('Delete')").click();
    }

    duplicate() {
        return cy.get(`${this.phaseSelector} .duplicate-phase`).click();
    }

    expectContainingTask(taskName) {
        cy.get('.task-header').should('contain', taskName);
    }

    expectGateTasksCountToBe(count) {
        cy.get(`${this.phaseSelector} .task-box .task-type-gatetask:visible`).should('have.length', count);
    }

    expectParallelGroupsCountToBe(count) {
        cy.get(`${this.phaseSelector} .task-box .task-type-parallelgroup:visible`).should('have.length', count);
    }

    expectActionsDisabled() {
        cy.get(`${this.phaseSelector} .duplicate-phase`).should('have.length', 0);
        cy.get(`${this.phaseSelector} .delete-phase`).should('have.length', 0);
        cy.get(`${this.phaseSelector} .task .duplicate-task`).should('have.length', 0);
        cy.get(`${this.phaseSelector} .task .delete-task`).should('have.length', 0);
        cy.get(`${this.phaseSelector} .task-generator .add-task`).should('have.length', 0);
        cy.get(`${this.phaseSelector} .display`).should('have.length', 0);
    }

    expectDuplicateEnabled() {
        cy.get(`${this.phaseSelector} .duplicate-phase`).should('have.length', 1);
    }

    expectDeleteDisabled() {
        cy.get(`${this.phaseSelector} .delete-phase`).should('have.length', 0);
    }
}


class TaskGenerator {
    constructor(usingElement) {
        this.usingElement = usingElement;
    }

    open() {
        cy.get(`${this.usingElement} .add-task`).click();
        return this;
    }

    close() {
        cy.get(`${this.usingElement} .cancel`).click();
        return this;
    }

    addTask(taskName, taskGroup, taskType) {
        cy.wait(750);
        cy.get(`${this.usingElement} .add-task`).click();
        cy.get(`${this.usingElement}`).find('.quick-title').type(taskName);
        cy.get(`${this.usingElement} .xlr-ctx-menu-toggle`).click();
        cy.get(`.xlr-ctx-menu-container li.xlr-ctx-menu-item[data-test='${taskGroup}']:visible`).trigger('mouseover');
        cy.get(`.xlr-ctx-menu-container .xlr-ctx-menu.active .xlr-ctx-menu-item[data-test='${taskType}'] > a`).scrollIntoView().click();
        return cy.get(`${this.usingElement} a:contains("Add"):visible`).click();
    }

    addManualTask(taskName) {
        return this.addTask(taskName, 'Core', 'xlrelease.Task');
    }

    addNotificationTask(taskName) {
        return this.addTask(taskName, 'Core', 'xlrelease.NotificationTask');
    }

    addScriptTask(taskName) {
        return this.addTask(taskName, 'Script', 'xlrelease.ScriptTask');
    }

    addGateTask(taskName) {
        return this.addTask(taskName, 'Core', 'xlrelease.GateTask');
    }

    addParallelGroup(taskName) {
        return this.addTask(taskName, 'Core', 'xlrelease.ParallelGroup');
    }

    expectTaskTypeDisabled(taskGroup, taskType) {
        this.open();
        cy.get(`${this.usingElement} .xlr-ctx-menu-toggle`).click();
        cy.get(`li.xlr-ctx-menu-item[data-test='${taskGroup}']:visible`).trigger('mouseover');
        cy.get(`li.xlr-ctx-menu-item[data-test='${taskType}']`).should('not.be.visible');
        cy.get(`${this.usingElement} .xlr-ctx-menu-toggle`).click();
        this.close();
        return this;
    }
}

class ReleaseTask {
    constructor(taskSelector) {
        this.taskSelector = taskSelector;
    }

    delete() {
        cy.get(`${this.taskSelector} .context-menu-button:first`).click();
        cy.get(`${this.taskSelector} a:contains('Delete')`).click({force: true});
        cy.get('.modal:visible .modal-header').should('contain', 'Delete Task');
        return cy.get(".modal:visible .button:contains('Delete')").click();
    }

    completeFromContextMenu() {
        cy.get(`${this.taskSelector} .context-menu-button:first`).click();
        cy.get(`${this.taskSelector} a:contains('Complete')`).click();
        cy.get('.modal:visible .modal-body').should('contain', 'complete the task');
        cy.get(`.modal:visible .button:contains('Complete')`).click();
        return this;
    }

    failFromContextMenu(comment) {
        cy.get(`${this.taskSelector} .context-menu-button:first`).click();
        cy.get(`${this.taskSelector} a:contains('Fail')`).click();
        cy.get('.modal:visible .modal-body').should('contain', 'fail the task');
        cy.get('textarea[mentio]:visible').type(comment);
        cy.get(`.modal:visible .button:contains('Fail')`).click();
        return this;
    }

    retryFromContextMenu(comment) {
        cy.get(`${this.taskSelector} .context-menu-button:first`).click();
        cy.get(`${this.taskSelector} a:contains('Retry')`).click();
        cy.get('.modal:visible .modal-body').should('contain', 'retry the task');
        cy.get('textarea[mentio]:visible').type(comment);
        cy.get(`.modal:visible .button:contains('Retry')`).click();
        return this;
    }

    expectActionToBeEnabled(action) {
        cy.get(`${this.taskSelector} .context-menu-button:first`).click();
        cy.get(`${this.taskSelector} a:contains('${action}')`).should('not.be.disabled');
        cy.get(".main-header .current-user").click();
        return this;
    }

    skipFromContextMenu(comment) {
        cy.get(`${this.taskSelector} .context-menu-button:first`).click();
        cy.get(`${this.taskSelector} a:contains('Skip')`).click();
        cy.get('.modal:visible .modal-body').should('contain', 'skip the task');

        cy.get('textarea[mentio]:visible').should('have.length', 1);
        cy.get('textarea[mentio]:visible').type(comment);
        cy.get(`.modal:visible .button:contains('Skip')`).click();
        return this;
    }

    expectNotFlagged() {
        cy.get(`${this.taskSelector} .flag-none-icon`).should('have.length', 0);
        return this;
    }

    expectFlaggedWith(flagName) {
        cy.get(`${this.taskSelector} .risk-flag-icon.${flagName}`).should('have.length', 1);
        return this;
    }

    complete() {
        this.selectMenuItem('Complete');
        cy.get('.modal:visible .modal-body').should('contain', 'complete the task');
        cy.get(".modal:visible .button:contains('Complete')").click();
        return this;
    }

    lock() {
        this.selectMenuItem('Lock');
        return this;
    }

    unlock() {
        this.selectMenuItem('Unlock');
        return this;
    }

    selectMenuItem(item) {
        this.openContextMenu();
        cy.get(`#context-menu-container a:contains('${item}')`).click();
        return this;
    }

    expectContextMenuItemVisible(item) {
        this.openContextMenu();
        cy.get(`#context-menu-container a:contains('${item}')`).scrollIntoView()
            .should('have.length',1);
        return this;
    }

    expectTaskStatusTooltipVisible(label, tooltipText) {
        this.openTaskStatusTooltip(label);
        cy.get('.tooltip')
            .should('be.visible')
            .and(($div) => {
                expect($div.text()).to.contain(tooltipText);
            });
        return this;
    }

    getNumberOfTasks() {
        return cy.get(`${this.taskSelector} .task`);
    }

    getTaskGenerator() {
        return new TaskGenerator(this.taskSelector);
    }

    openContextMenu() {
        cy.get(`${this.taskSelector} .context-menu-button`).should('have.length', 1);
        cy.get(`${this.taskSelector} .context-menu-button`).click();
        cy.get("#context-menu-container").should('have.length', 1);
        return this;
    }

    openTaskStatusTooltip(label) {
        cy.get(`${this.taskSelector} ${label}`).then(function ($status) {
            $status.trigger('mouseover');
        });
        return this;
    }

    openChangeTypeMenu() {
        cy.get(`.dropdown-submenu[data-content-id="Change type"]`).should('have.length', 1);
        cy.doWithRetry(
            () => cy.get(`.dropdown-submenu[data-content-id="Change type"]`).then(function ($heading) {
                $heading.trigger('mouseover');
            }),
            ('ul.dropdown-menu[data-id="Change type"]')
        );
        return this;
    }

    openSubmenu(menuId) {
        cy.get(`.dropdown-submenu[data-content-id="${menuId}"]`).should('have.length', 1);
        cy.doWithRetry(
            () => cy.get(`.dropdown-submenu[data-content-id="${menuId}"]`).then(function ($heading) {
                $heading.trigger('mouseover');
            }),
            (`ul.dropdown-menu[data-id="${menuId}"]`)
        );
        return this;
    }

    changeTaskTypeTo(newType) {
        cy.get(`ul.dropdown-menu li a:contains('${newType}')`).click({force:true});
        cy.get('.modal-footer button.primary').click();
        return this;
    }

    closeContextMenu() {
        cy.get("#context-menu-container").should('have.length', 1);
        cy.get(`${this.taskSelector} .context-menu-button`).click();
        return this;
    }

    expectTaskFailureHandlerEnabled() {
        cy.get(`${this.taskSelector} span.failure-handler-icon`).should('be.visible');
        return this;
    }

    expectTaskFailureHandlerDisabled() {
        cy.get(`${this.taskSelector} span.failure-handler-icon`).should('not.be.visible');
        return this;
    }

    expectStartsDateTimeToBe(formattedDate, formattedTime) {
        cy.get(`${this.taskSelector} span:contains("${formattedDate} at ${formattedTime}")`).should('be.visible');
        // expect(element(By.$(`${this.task} span:contains("${formattedDate} at ${formattedTime}")`))).toBePresent();
        return this;
    }

    expandNestedTasks() {
        cy.get(`${this.taskSelector} .task-group .link`).click();
        return this;
    }

    getTask(taskName) {
        return new ReleaseTask(`${this.taskSelector} .task:contains('${taskName}')`);
    }

    duplicate() {
        cy.get(`${this.taskSelector} .context-menu-button:first`).click();
        cy.get(`${this.taskSelector} a:contains('Duplicate')`).click({force: true});
        cy.get('#context-menu-container').should('have.length', 0);
    }

    expectDuplicateEnabled() {
        cy.get(`${this.taskSelector} .context-menu-button:first`).click();
        cy.get(`${this.taskSelector} .duplicate-task-button[disabled=disabled]`).should('have.length', 0);
        return cy.get(".main-header .current-user").click();
    }

    expectDeleteDisabled() {
        cy.get(`${this.taskSelector} .context-menu-button:first`).click();
        cy.get(`${this.taskSelector} .remove-task-button[disabled=disabled]`).should('have.length', 1);
        return cy.get(".main-header .current-user").click();
    }

    expectDisabledContextMenuItem(item) {
        cy.get(`#context-menu-container li:contains('${item}')`).should('have.class', 'disabled');
        return this;
    }

    expectNoChangeTypeSubMenu() {
        cy.get('.dropdown-submenu').should('not.be.visible');
        return this;
    }

    expectEnabledContextMenuItem(item) {
        this.openContextMenu();
        cy.get(`#context-menu-container a:contains('${item}')`).should('have.length', 1);
        cy.get(`#context-menu-container a:contains('${item}')[disabled]`).should('have.length', 0);
        return this;
    }

    expectToBeLocked() {
        cy.get(`${this.taskSelector}`)
            .closest('.task-box')
            .should('have.class', 'locked');
        return this;
    }

    expectToBeUnlocked() {
        cy.get(`${this.taskSelector}`)
            .closest('.task-box')
            .should('not.have.class', 'locked');
        return this;
    }

    expectTitleToBe(title) {
        cy.get(`${this.taskSelector} .task-title`).should('contain', title);
        return this;
    }

    expectToBeInProgress() {
        cy.get(`${this.taskSelector} .label-inprogress`).should('have.length', 1);
        return this;
    }

    expectStatusLine(title, link) {
        cy.get(`${this.taskSelector} .status-line a:contains('${title}')`).should('be.visible');
        cy.get(`${this.taskSelector} .status-line a`).should('have.attr','href').and('contains', link);
        return this;
    }
}

class Teams {
    constructor() {
        cy.get('#teams-permissions-table').should('be.visible');
    }

    refresh() {
        cy.reload();
        return new Teams();
    }

    expectTeamExists(teamName) {
        // expect(element(`#teams-permissions-table .text-column:contains('${teamName}')`).count()).toBe(1);
        cy.get(`#teams-permissions-table .text-column:contains('${teamName}')`).should('have.length',1);
        return this;
    }

    expectTeamContains(teamName, ...members) {
        const teamEditor = new TagEditor(`tr:contains('${teamName}')`);
        for (const member of members) {
            teamEditor.expectTaggedWith(`${member}`);
        }
        return this;
    }
}
