package controllers

import java.util.concurrent.{ExecutorService, Executors}

import akka.actor.{ActorRef, ActorSystem}
import akka.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import akka.pattern.ask
import akka.util.Timeout
import com.xebialabs.xlrelease.stress.domain.Task
import com.xebialabs.xlrelease.stress.runner.ActorRunner
import com.xebialabs.xlrelease.stress.runner.Protocol.{Busy, GetStatus, InvalidParameters, ListScenarios, RunResponse, RunScenario, RunnerStatus, ScenarioList, Started, UnknownScenario}
import javax.inject.{Inject, Singleton}
import play.api.Configuration
import play.api.libs.json.Json.toJsObject
import play.api.libs.json._
import play.api.mvc.{AbstractController, Action, AnyContent, ControllerComponents}

import scala.concurrent.ExecutionContext
import scala.concurrent.duration._
import scala.language.postfixOps

@Singleton
class RunScenarioController @Inject()(cc: ControllerComponents,
                                      config: Configuration,
                                      system: ActorSystem) extends AbstractController(cc) with SprayJsonSupport {

  lazy val actorRunner: ActorRef = system.actorOf(ActorRunner.props(config))

  lazy val singleScenarioClient: ExecutorService = Executors.newSingleThreadExecutor()
  implicit lazy val ec: ExecutionContext = ExecutionContext.fromExecutor(singleScenarioClient)

  implicit val timeout: Timeout = 5 seconds
  val defaultSetupTimeout = 1 minute
  val defaultProgramTimeout = 10 minutes

  def run(scenario: String): Action[JsValue] = Action.async(parse.json) { request =>
    val setupTimeout = request.getQueryString("setupTimeout").fold(defaultSetupTimeout)(s => Integer.parseInt(s) seconds)
    val programTimeout = request.getQueryString("programTimeout").fold(defaultProgramTimeout)(s => Integer.parseInt(s) seconds)
    val taskId = request.getQueryString("taskId").map(Task.ID.apply).get
    (actorRunner ? RunScenario(scenario, request.body, taskId, setupTimeout, programTimeout))
      .mapTo[RunResponse].map {
        case resp: Started =>
          Ok(toJsObject(resp))
        case resp: Busy =>
          ServiceUnavailable(toJsObject(resp))
        case resp: UnknownScenario =>
          NotFound(toJsObject(resp))
        case resp: InvalidParameters =>
          BadRequest(toJsObject(resp))
      }
  }

  def get(): Action[AnyContent] = Action.async { _ =>
    (actorRunner ? GetStatus).mapTo[RunnerStatus].map(status =>
      Ok(toJsObject(status))
    )
  }

  def list(): Action[AnyContent] = Action.async { _ =>
    (actorRunner ? ListScenarios).mapTo[ScenarioList].map(scenario =>
      Ok(toJsObject(scenario))
    )
  }

}
