package com.xebialabs.xlrelease.stress.utils

import java.io.{File, FileInputStream, FileOutputStream, InputStream}
import java.util.zip.{ZipEntry, ZipInputStream, ZipOutputStream}

import com.xebialabs.xlrelease.stress.utils.ResourceManagement._
import org.apache.commons.io.IOUtils

import scala.util.Try


object ZipUtils {
  def processEntries[B](zipFile: InputStream)(doStuff: ZipInputStream => ZipEntry => B): Seq[B] = {
    using(new ZipInputStream(zipFile)) { zipInputStream =>
      val doThing = doStuff(zipInputStream)
      Stream.continually(Try(zipInputStream.getNextEntry).getOrElse(null))
        .takeWhile(_ != null) // while not EOF and not corrupted
        .map(doThing)
        .force
    }
  }

  def extract(zipFile: InputStream, dest: File): Unit = collect(zipFile, dest) { case _ => true }

  def collect[A](zipFile: InputStream, dest: File)(body: PartialFunction[String, A]): Seq[A] = {
    dest.mkdirs()
    processEntries(zipFile) { zis => entry =>
      val fileName = entry.getName
      body.lift(fileName).map { result =>
        val newFile = new File(dest, fileName)
        new File(newFile.getParent).mkdirs()
        if (entry.isDirectory) {
          newFile.mkdir()
        } else {
          using(new FileOutputStream(newFile)) { fos =>
            IOUtils.copy(zis, fos)
          }
        }
        result
      }
    }.flatten
  }

  // Note: if you have files persisted on a filesystem, do use archive() method instead.
  def create(files: Map[String, InputStream], dest: File): File = {
    val out = new ZipOutputStream(new FileOutputStream(dest))
    files.foreach { case (filename, content) =>
      val entry = new ZipEntry(filename)
      out.putNextEntry(entry)
      IOUtils.copy(content, out)
      content.close()
      out.closeEntry()
    }
    out.close()
    dest
  }

  def archive(files: Seq[File], dest: File, stripPrefix: Option[String] = None): File = {
    def filename(file: File) = stripPrefix.map(file.getPath.stripPrefix(_).stripPrefix(File.separator)).getOrElse(file.getAbsolutePath)
    val out = new ZipOutputStream(new FileOutputStream(dest))
    files.foreach { file =>
      val entry = new ZipEntry(filename(file))
      out.putNextEntry(entry)
      ResourceManagement.using(new FileInputStream(file)) { content =>
        IOUtils.copy(content, out)
      }
      out.closeEntry()
    }
    out.close()
    dest
  }
}
