package com.xebialabs.xlrelease.stress.utils

import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.Scenario
import com.xebialabs.xlrelease.stress.domain.{Comment, Release, TaskStatus, Template}
import org.joda.time.DateTime

import scala.concurrent.duration._
import scala.language.postfixOps
import scala.util.matching.Regex

trait ScenarioUtils { scenario: Scenario =>

  protected val idFromCommentRegex: Regex =
    "Created (release|template) \\[.*\\]\\(#\\/(releases|templates)\\/(.*)\\).".r

  def getIdFromComment(str: String): Option[Template.ID] =
    idFromCommentRegex.findFirstMatchIn(str).flatMap { m =>
      if (m.groupCount < 3) {
        None
      } else {
        Some(Template.ID(m.group(3).replaceAll("-", "/")))
      }
    }

  def createReleaseFromGroovy(title: String,
                              groovyScript: String,
                              replacements: Map[String, String]): IO[Release.ID] = {
    val interpolatedScript = replacements.foldLeft(groovyScript) { case (script, (placeholder, value)) =>
        script.replaceAll(placeholder, value)
    }

    api.xlr.users.admin() >>= { implicit session =>
      for {
        _ <- api.log.debug(s"createReleaseFromGroovy($title)")
        phaseId <- api.xlr.releases.create(title)
        taskId <- api.xlr.tasks.appendScript(phaseId, "groovy task", "xlrelease.GroovyScriptTask", interpolatedScript)
        manualTaskId <- api.xlr.tasks.appendManual(phaseId, "wait before completing")
        _ <- api.xlr.tasks.assignTo(manualTaskId, session.user.username)
        _ <- api.xlr.releases.start(phaseId.release)
        _ <- api.xlr.tasks.waitFor(taskId, TaskStatus.Completed, interval = 5 seconds, retries = Some(20))
        comments <- api.xlr.tasks.getComments(taskId)
        comment <- comments.lastOption.map(_.pure[IO]).getOrElse {
          IO.raiseError[Comment](new RuntimeException(s"No comments in task ${taskId.show}"))
        }
        _ <- api.log.debug("last comment: "+ comment)
        releaseId <- getIdFromComment(comment.text).map(_.pure[IO]).getOrElse {
          IO.raiseError(new RuntimeException(s"Cannot extract releaseId from comment: $comment"))
        }
        _ <- api.log.debug(s"releaseId from comment: $releaseId")
        _ <- api.xlr.tasks.complete(manualTaskId, comment = None)
//        _ <- api.xlr.releases.waitFor(phaseId.release, status = ReleaseStatus.Completed, interval = 5 seconds, retries = Some(10))
//        _ <- api.log.info(s"createReleaseFromGroovy($title) completed: $releaseId")
      } yield Release.ID(releaseId.id)
    }
  }

  // TODO: move to lib?
  protected def withHealthCheck[A, B](program: IO[A], checkInterval: FiniteDuration, checkIO: IO[B]): IO[(A, List[(DateTime, FiniteDuration, B)])] =
    api.control.backgroundOf(program) {
      for {
        res <- healthCheck(checkIO)
        _ <- api.control.sleep(checkInterval)
      } yield res
    }

  // TODO: move to lib?
  protected def healthCheck[A](program: IO[A]): IO[(DateTime, FiniteDuration, A)] =
    for {
      start <- api.control.now()
      result <- api.control.time(program)
    } yield (start, result._1, result._2)

  // TODO: move to control lib
  def rampUp[A](start: Int, end: Int, step: Int => Int)(program: Int => IO[A]): IO[List[List[A]]] = {
    RampUpRange.toList(RampUpRange(start, end, step)).map { n =>
      api.control.parallel[A](n) { i =>
        program(i)
      }
    }.sequence
  }

  def grouped[A](howMany: Int, groupSize: Int)(program: Int => IO[A]): IO[List[List[A]]] = {
    (0 until howMany).grouped(groupSize).toList.map { batch =>
      api.control.parallel(groupSize) { i =>
        program(batch.head + i)
      }
    }.sequence
  }

  case class RampUpRange(start: Int, end: Int, step: Int => Int = _ + 1)

  object RampUpRange {
    def toStream(range: RampUpRange): Stream[Int] =
      if (range.start <= range.end) {
        range.start #:: toStream(range.copy(start = range.step(range.start)))
      } else Stream.empty

    def toList(range: RampUpRange): List[Int] = toStream(range).toList
  }
}
