package com.xebialabs.xlrelease.stress.scenarios.meta

import cats.Show
import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.{API, Scenario}

object ScenarioWithBackground {

  case class MakeScenarioWithBackground[A <: Scenario, PA, B <: Scenario, PB](mkFgScenario: Scenario.ScenarioMaker[A, PA],
                                                                              mkBgScenario: Scenario.ScenarioMaker[B, PB])
    extends Scenario.ScenarioMaker[ScenarioWithBackground[A, B], (PA, PB)](s"${mkFgScenario.name}_withBackground_${mkBgScenario.name}") {

    import play.api.libs.functional.syntax._
    import play.api.libs.json._

    implicit val readParameters: Reads[(PA, PB)] = {
      (
        (JsPath \ "foreground").read[PA](mkFgScenario.readParameters) and
          (JsPath \ "background").read[PB](mkBgScenario.readParameters)
        ) ((fg, bg) => (fg, bg))
    }

    def scenario(parameters: (PA, PB))(implicit api: API): ScenarioWithBackground[A, B] =
      ScenarioWithBackground[A, B](mkFgScenario.scenario(parameters._1), mkBgScenario.scenario(parameters._2))
  }

}

case class ScenarioWithBackground[A <: Scenario, B <: Scenario](foreground: A, background: B)
                                                               (implicit val api: API)
  extends Scenario {

  foreground.parent = Some(this)
  background.parent = Some(this)

  type Params = (foreground.Params, background.Params)

  def name: String = s"ScenarioWithBackground(${foreground.name}, ${background.name})"

  def setup: IO[Params] = {
    // login before parallel setup, in order to avoid dreaded:
    // java.sql.SQLIntegrityConstraintViolationException: Duplicate entry 'admin' for key 'PRIMARY'
    api.xlr.users.admin() >> api.control.fork(
      setupScenario("foreground", foreground),
      setupScenario("background", background)
    )
  }

  def program(params: Params): IO[Unit] =
    api.control.backgroundOf(foreground.program(params._1))(background.program(params._2)) >> api.control.nop

  def cleanup(params: Params): IO[Unit] =
    api.control.fork(
      api.log.info("Cleaning up foreground scenario") >> foreground.cleanup(params._1) >> api.log.info("Foreground scenario: cleanup complete"),
      api.log.info("Cleaning up background scenario") >> background.cleanup(params._2) >> api.log.info("Background scenario: cleanup complete")
    ) >> api.control.nop

  implicit val showParams: Show[Params] = {
    case (fgParams, bgParams) =>
      s"(${foreground.showParams.show(fgParams)}, ${background.showParams.show(bgParams)})"
  }

  private def setupScenario[S <: Scenario](name: String, scenario: S): IO[S#Params] =
    for {
      _ <- api.log.info(s"Setting up $name scenario (${scenario.name})")
      p <- scenario.setup
      _ <- api.log.info(s"Setup of $name scenario (${scenario.name}) complete")
    } yield p

}
