package com.xebialabs.xlrelease.stress.scenarios

import akka.http.scaladsl.model.{HttpResponse, StatusCode, StatusCodes}
import cats.Show
import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.api.metric.MetricFunction
import com.xebialabs.xlrelease.stress.domain.Configuration
import com.xebialabs.xlrelease.stress.utils.HttpHelpers._
import com.xebialabs.xlrelease.stress.{API, Scenario}
import play.api.libs.functional.syntax._
import play.api.libs.json._

import scala.concurrent.duration._
import scala.language.postfixOps
object WebhooksRequestsBaselineScenario
  extends Scenario.ScenarioMaker[WebhooksRequestsBaselineScenario, (Int, Int, String, JsObject)]("webhooks.baseline") {

  override val defaultParameters: JsObject = Json.obj(
    "concurrency" -> 1,
    "numberOfRequests" -> 256,
    "endpointPath" -> "webhook_perf_test",
    "payload" -> Json.obj()
  )

  implicit val readParameters: Reads[(Int, Int, String, JsObject)] = (
    (JsPath \ "concurrency").read[Int] and
      (JsPath \ "numberOfRequests").read[Int] and
      (JsPath \ "endpointPath").read[String] and
      (JsPath \ "payload").read[JsObject]
    ) ((a, b, c, d) => (a, b, c, d))

  def scenario(parameters: (Int, Int, String, JsObject))(implicit api: API): WebhooksRequestsBaselineScenario = parameters match {
    case (a, b, c, d) => WebhooksRequestsBaselineScenario(a, b, c, d)
  }
}

case class WebhooksRequestsBaselineScenario(concurrency: Int, numberOfRequests: Int, endpointPath: String, payload: JsObject)
                                           (implicit val api: API) extends Scenario with spray.json.DefaultJsonProtocol {
  type Params = Configuration.ID

  def name: String = "WebhooksRequestsBaseline"

  def setup: IO[Params] =
    api.xlr.users.admin() >>= { implicit session =>
      api.xlr.configurations.create(
        Configuration.postWebhook(endpointPath, "test webhook for performance")
      )
    }

  lazy val perStatusResponseTimes: Map[StatusCode, MetricFunction] = Seq[StatusCode](
    StatusCodes.OK,
    StatusCodes.NotFound,
    StatusCodes.InternalServerError,
    StatusCodes.Unauthorized
  ).map { code =>
    code -> api.metric.named(s"webhooks baseline response time: ${code.toString}")
  }.toMap
  lazy val otherResponseTimes: MetricFunction = api.metric.named("webhooks baseline response time: others")
  lazy val allResponseTimes: MetricFunction = api.metric.named("webhooks baseline reponse time: ALL")

  def request: IO[HttpResponse] =
    for {
      resp <- api.http.post(api.xlrConfig.server.root(_ ?/ "webhooks" / endpointPath),
        payload.toHttpEntity
      )
      _ <- api.http.discard(resp)
    } yield resp

  def measuredRequest: IO[(FiniteDuration, StatusCode)] =
    for {
      res <- api.control.time(request)
      (time, resp) = res
      metric = perStatusResponseTimes.getOrElse(resp.status, otherResponseTimes)
      _ <- metric.add(time)
      _ <- allResponseTimes.add(time)
      _ <- api.log.debug(s"${resp.status}: ${time.toMillis}ms")
    } yield time -> resp.status

  def program(params: Params): IO[Unit] =
    for {
      _ <- api.log.info("START")
      result <- api.control.time {
        api.control.concurrently(concurrency) {
          (0 until numberOfRequests).map { _ =>
            measuredRequest
          }
        }
      }
      (totalTime, measuredRequests) = result
      numRequests = measuredRequests.flatten.length
      _ <- if (totalTime.toMillis > 0 && numRequests > 0) {
        val reqPerMinute: Double = ((numRequests * 60000) / totalTime.toMillis)
        api.log.info(s"sent $numRequests in $totalTime: $reqPerMinute/min (concurrency: $concurrency)")
      } else {
        api.log.warn(s"infinite rate? ${numRequests} in no time")
      }
      _ <- api.log.info("END")
    } yield ()

  def cleanup(params: Params): IO[Unit] =
    api.xlr.users.admin() >>= { implicit session =>
      api.xlr.configurations.delete(params)
    }

  implicit val showParams: Show[Params] = Show[Params](identity[String])
}
