package com.xebialabs.xlrelease.stress.scenarios

import cats._
import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.domain.Target.VariableTarget
import com.xebialabs.xlrelease.stress.domain._
import com.xebialabs.xlrelease.stress.protocol.CreateReleaseArgs
import com.xebialabs.xlrelease.stress.{API, Scenario}
import spray.json._

import scala.concurrent.duration._
import scala.language.postfixOps

// TODO: use folders

object SubReleases
  extends Scenario.ScenarioMaker[SubReleases, (Int, Int, Int, Int, FiniteDuration, Int)]("sub.releases") {

  import play.api.libs.functional.syntax._
  import play.api.libs.json._

  override val defaultParameters = Json.obj(
    "numMasterReleases" -> 10,
    "numPhases" -> 10,
    "numCreateReleaseTasks" -> 8,
    "numAutomatedTasks" -> 8,
    "sleepTimeout" -> 8,
    "concurrency" -> 8
  )

  implicit val readParameters: Reads[(Int, Int, Int, Int, FiniteDuration, Int)] = (
    (JsPath \ "numMasterReleases").read[Int] and
      (JsPath \ "numPhases").read[Int] and
      (JsPath \ "numCreateReleaseTasks").read[Int] and
      (JsPath \ "numAutomatedTasks").read[Int] and
      (JsPath \ "sleepTimeout").read[Int].map(_ seconds) and
      (JsPath \ "concurrency").read[Int]
    ) ((a, b, c, d, e, f) => (a, b, c, d, e, f))

  def scenario(parameters: (Int, Int, Int, Int, FiniteDuration, Int))(implicit api: API): SubReleases = parameters match {
    case (a, b, c, d, e, f) =>
      SubReleases(
        numMasterReleases = a,
        numPhases = b,
        numCreateReleaseTasks = c,
        numAutomatedTasks = d,
        sleepTimeout = e,
        concurrency = f
      )
  }

  val gateTaskTitle = "Wait for Sub-releases"
  val manualTaskTitle = "Complete Me"
}

case class SubReleases(numMasterReleases: Int,
                       numPhases: Int,
                       numCreateReleaseTasks: Int,
                       numAutomatedTasks: Int,
                       sleepTimeout: FiniteDuration,
                       concurrency: Int)
                      (implicit val api: API)
  extends Scenario with DefaultJsonProtocol {
  type Params = (Template.ID, Template.ID)

  def name: String = "SubReleases"

  lazy val reachManualTaskTime = api.metric.named("reachManualTask")
  lazy val completeManualTaskTime = api.metric.named("completeManual")

  def setup: IO[Params] =
    api.xlr.users.admin() >>= { implicit session =>
      for {
        slaveTemplateId <- setupSlaveTemplate()
        masterTemplateId <- setupMasterTemplate(slaveTemplateId)
      } yield (masterTemplateId, slaveTemplateId)
    }

  def program(params: Params): IO[Unit] =
    api.xlr.users.admin() >>= { implicit session =>
      api.control.concurrently(concurrency) {
        (1 to numMasterReleases).map { i =>
          for {
            manualTaskId <- reachManualTaskTime.measure(
              api.control.time(runMasterRelease(masterTemplateId = params._1)(i))
            )
            runningReleaseJson <- api.xlr.releases.get(manualTaskId.phaseId.release)
            _ <- api.log.session.info("*** Release JSON just AFTER starting GATE TASK:")
            _ <- api.log.session.info(runningReleaseJson.prettyPrint)
            _ <- api.log.session.info("***")
            _ <- completeManualTaskTime.measure(
              api.control.time(api.xlr.tasks.complete(manualTaskId, Some("completed")))
            )
          } yield ()
        }
      } >> api.control.nop
    }

  def runMasterRelease(masterTemplateId: Template.ID)
                      (i: Int)
                      (implicit session: User.Session): IO[Task.ID] =
    for {
      _ <- api.log.session.info(s"Creating master release #$i")
      releaseId <- api.xlr.releases.createFromTemplate(masterTemplateId, CreateReleaseArgs(
        title = s"Master-$i",
        variables = Map.empty,
        autoStart = false
      ))
      releaseJson <- api.xlr.releases.get(releaseId)
      _ <- api.log.info("### Master Release JSON:")
      _ <- api.log.session.info(releaseJson.prettyPrint)
      _ <- api.log.info("###")
      preGateTaskIds <- api.xlr.releases.getTasksByTitle(releaseId, taskTitle = "Get ready!")
      preGateTaskId <- preGateTaskIds.headOption match {
        case None => api.control.fail("Cannot find preGate task id")
        case Some(id) => api.control.ok(id)
      }
      manualTaskIds <- api.xlr.releases.getTasksByTitle(releaseId, taskTitle = SubReleases.manualTaskTitle)
      manualTaskId <- manualTaskIds.headOption match {
        case None => api.control.fail("Cannot find manual task id")
        case Some(id) => api.control.ok(id)
      }
      _ <- api.xlr.releases.start(releaseId)
      _ <- api.xlr.tasks.waitFor(preGateTaskId, TaskStatus.InProgress, retries = None)
      runningReleaseJson <- api.xlr.releases.get(releaseId)
      _ <- api.log.session.info("*** Release JSON just BEFORE starting GATE TASK:")
      _ <- api.log.session.info(runningReleaseJson.prettyPrint)
      _ <- api.log.session.info("***")
      _ <- api.xlr.tasks.complete(preGateTaskId)
      _ <- api.xlr.tasks.waitFor(manualTaskId, TaskStatus.InProgress, retries = None)
    } yield manualTaskId

  def cleanup(params: Params): IO[Unit] =
    api.xlr.users.admin() >>= { implicit session =>
      api.xlr.templates.delete(params._1) >>
        api.xlr.templates.delete(params._2)
    }

  implicit val showParams: Show[Params] = Show[Params] {
    case (masterTemplateId: Template.ID, slaveTemplateId: Template.ID) =>
      s"master: ${masterTemplateId.show}, slave: ${slaveTemplateId.show}"
  }

  private def setupSlaveTemplate()
                                (implicit session: User.Session): IO[Template.ID] =
    for {
      slavePhaseId <- api.xlr.templates.create("Slave template", scriptUser = Some(session.user), tags = List("slave"))
      _ <- api.control.sequenced(numAutomatedTasks) { i =>
        api.xlr.tasks.appendScript(slavePhaseId, s"Auto-$i", "xlrelease.ScriptTask",
          s"""|from time import sleep
              |
              |print("Waiting ${sleepTimeout.toSeconds} seconds...\\n\\n")
              |sleep(${sleepTimeout.toSeconds})
              |print("Done.\\n\\n")
              |""".stripMargin
        )
      }
    } yield Template.ID(slavePhaseId.release.id)

  private def setupMasterTemplate(slaveTemplateId: Template.ID)
                                 (implicit session: User.Session): IO[Template.ID] =
    for {
      masterPhaseId <- api.xlr.templates.create("Master template", scriptUser = Some(session.user), tags = List("master"))
      firstPhase <- fillPhase(slaveTemplateId, masterPhaseId, 0)
      nextPhases <- api.control.sequenced(numPhases - 1) { phaseIndex =>
        createAndFillPhase(masterPhaseId.release, slaveTemplateId, 1 + phaseIndex)
      }
      allSubReleaseTargets = firstPhase ++ nextPhases.flatten
      gatePhaseId <- api.xlr.phases.appendPhase(masterPhaseId.release)
      preGateTaskId <- api.xlr.tasks.appendManual(gatePhaseId, "Get ready!")
      _ <- api.xlr.tasks.assignTo(preGateTaskId, session.user.username)
      _ <- api.xlr.tasks.appendGate(gatePhaseId, SubReleases.gateTaskTitle, allSubReleaseTargets)
      manualTaskId <- api.xlr.tasks.appendManual(gatePhaseId, SubReleases.manualTaskTitle)
      _ <- api.xlr.tasks.assignTo(manualTaskId, session.user.username)
    } yield Template.ID(masterPhaseId.release.id)


  private def createAndFillPhase(masterTemplateId: Release.ID, slaveTemplateId: Template.ID, phaseIndex: Int)
                                (implicit session: User.Session): IO[List[VariableTarget]] =
    api.xlr.phases.appendPhase(masterTemplateId) >>= (phaseId => fillPhase(slaveTemplateId, phaseId, phaseIndex))

  private def fillPhase(slaveTemplateId: Template.ID, phaseId: Phase.ID, phaseIndex: Int)
                       (implicit session: User.Session): IO[List[VariableTarget]] =
    api.control.sequenced(numCreateReleaseTasks) {
      appendCreateReleaseTask(slaveTemplateId, phaseId, phaseIndex)
    }

  private def appendCreateReleaseTask(slaveTemplateId: Template.ID, phaseId: Phase.ID, phaseIndex: Int)
                                     (i: Int)
                                     (implicit session: User.Session): IO[VariableTarget] = {
    val variableName = s"slave_${phaseIndex}_$i"
    val variableKey = "${" + variableName + "}"
    val taskJson: JsObject = JsObject(
      "id" -> JsNull,
      "type" -> "xlrelease.CreateReleaseTask".toJson,
      "title" -> s"CRslave_${phaseIndex}_$i".toJson,
      "startRelease" -> true.toJson,
      "templateId" -> slaveTemplateId.show.toJson,
      "newReleaseTitle" -> s"Slave-$phaseIndex-$i".toJson,
      "createdReleaseId" -> variableKey.toJson,
      "releaseTags" -> List("slave").toJson
    )
    for {
      variableId <- api.xlr.releases.createVariable(phaseId.release, Variable.string(
        id = None,
        key = variableName,
        value = None,
        required = false,
        showOnReleaseStart = false
      ))
      _ <- api.xlr.tasks.append(taskJson, Target.PhaseTarget(phaseId))
    } yield VariableTarget(variableId, variableName)
  }
}
