package com.xebialabs.xlrelease.stress.scenarios

import cats._
import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.domain.ReleaseStatus.{Completed, InProgress}
import com.xebialabs.xlrelease.stress.domain.Task.{MarkTrackedItems, RegisterTrackedItems, WaitForTrackedItems}
import com.xebialabs.xlrelease.stress.domain._
import com.xebialabs.xlrelease.stress.scenarios.ReleasesPerDeliveryScenario.PerDeliveryScenarioParams
import com.xebialabs.xlrelease.stress.utils.ScenarioUtils
import com.xebialabs.xlrelease.stress.{API, Scenario}
import play.api.libs.functional.syntax._
import play.api.libs.json.{JsPath, Json, Reads}
import spray.json._

import scala.concurrent.duration._
import scala.language.postfixOps

object ReleasesPerDeliveryScenario
  extends Scenario.ScenarioMaker[ReleasesPerDeliveryScenario, (Int, Int, Int)]("multiReleases.delivery") {

  override val defaultParameters = Json.obj(
    "numComponentReleases" -> 10,
    "numStagesPerPattern" -> 10,
    "concurrency" -> 8
  )

  implicit val readParameters: Reads[(Int, Int, Int)] = (
    (JsPath \ "numComponentReleases").read[Int] and // this will be same as number of tracked items considering 1 tracked item per release
      (JsPath \ "numStagesPerPattern").read[Int] and
      (JsPath \ "concurrency").read[Int]
    ) ((a, b, c) => (a, b, c))

  def scenario(parameters: (Int, Int, Int))(implicit api: API): ReleasesPerDeliveryScenario = parameters match {
    case (a, b, c) =>
      ReleasesPerDeliveryScenario(
        numComponentReleases = a,
        numStagesPerPattern = b,
        concurrency = c
      )
  }

  case class PerDeliveryScenarioParams(folder: Folder,
                                       stages: Map[String, Stage.ID],
                                       deliveriesToReleases: Map[Delivery.ID, List[(Task.ID, Release.ID)]],
                                       masterReleaseIds: List[Release.ID])

}

case class ReleasesPerDeliveryScenario(
                                        numComponentReleases: Int,
                                        numStagesPerPattern: Int,
                                        concurrency: Int)
                                      (implicit val api: API) extends Scenario with ScenarioUtils with DefaultJsonProtocol {
  override type Params = PerDeliveryScenarioParams

  lazy val reachMasterReleaseCompletion = api.metric.named("reachMasterReleaseCompletion")

  def name: String = "ActiveDeliveriesScenario"

  def setup: IO[PerDeliveryScenarioParams] =
    api.xlr.users.admin() >>= { implicit session =>
      for {
        _ <- api.log.session.debug("creating parent folder")
        start <- api.control.now()
        folderName = s"deliveries-${start.toString("YYYY-MM-dd_HH:mm:ss")}"
        folder <- api.xlr.folders.createFolder(Folder(folderName))
        stageId <- api.xlr.patterns.create(s"Pattern", folder.id.show)
        patternId = stageId.patternId
        _ <- api.xlr.patterns.deleteStage(stageId)
        //        _ <- api.control.sequenced(numComponentReleases) { i => api.xlr.patterns.addTrackedItem(s"ti$i", patternId) }
        stages <- api.control.sequenced(numStagesPerPattern) { i =>
          api.xlr.patterns.addStage(s"Stage-$i", patternId, i).map {
            stageId => s"s$i" -> stageId
          }
        }.map(_.toMap)
        deliveriesToReleases <- api.control.sequenced(numComponentReleases) { i =>
          for {
            deliveryId <- api.xlr.deliveries.createFromPattern(s"Delivery-$i", patternId, folder.id)
            taskAndReleaseIds <- api.control.sequenced(numComponentReleases) {
              createComponentRelease(deliveryId, s"Delivery-$i", stages)
            }
          } yield deliveryId -> taskAndReleaseIds
        }.map(_.toMap)
        masterReleaseIds <- createMasterReleases(deliveriesToReleases.keys.toList)
      } yield PerDeliveryScenarioParams(folder, stages, deliveriesToReleases, masterReleaseIds)
    }

  def program(params: PerDeliveryScenarioParams): IO[Unit] =
    api.xlr.users.admin() >>= { implicit session =>
      for {
        _ <- api.log.info(s"started program for multi.deliveries with $numComponentReleases deliveries per pattern")
        _ <- reachMasterReleaseCompletion.measure(api.control.time(
          for {
            manualTaskIds <- api.control.concurrently(concurrency) {
              params.deliveriesToReleases.values.flatten.map {
                case (taskId, releaseId) =>
                  for {
                    _ <- api.xlr.releases.start(releaseId)
                    _ <- api.xlr.tasks.waitFor(taskId, TaskStatus.InProgress, 1 second)
                  } yield taskId
              }
            }.map(_.flatten)
            _ <- api.control.concurrently(concurrency) {
              params.masterReleaseIds.map(releaseId =>
                api.xlr.releases.start(releaseId) >>
                  api.xlr.releases.waitFor(releaseId, InProgress, interval = 1 second)) // comment to get race condition
            }
            _ <- api.control.concurrently(concurrency) {
              manualTaskIds.map { taskId =>
                api.xlr.tasks.complete(taskId)
              }
            }
            _ <- api.control.concurrently(concurrency) {
              params.masterReleaseIds.map(api.xlr.releases.waitFor(_, Completed, interval = 1 second))
            }
          } yield ()
        ))
      } yield ()
    }

  def cleanup(params: PerDeliveryScenarioParams): IO[Unit] =
    api.log.info(s"check cleanup") >>
      api.xlr.users.admin() >>= { implicit session =>
      api.xlr.folders.deleteFolder(params.folder.id)
    }

  implicit val showParams: Show[PerDeliveryScenarioParams] = Show[PerDeliveryScenarioParams] {
    params => s"folder: ${params.folder.id.show}"
  }

  def createMasterReleases(deliveryIds: List[Delivery.ID])(implicit session: User.Session): IO[List[Release.ID]] =
    for {
      phaseId <- api.xlr.releases.create(s"Master Release - Tracked Items Per Phase", Some(session.user))
      trackedItems = (0 until numComponentReleases).map(i => s"ti$i")
      _ <- api.xlr.tasks.appendDeliveryTask(phaseId, WaitForTrackedItems(s"wait all items in ${deliveryIds.head.id}",
        deliveryIds.head.id, trackedItems.toList, s"Stage-${numStagesPerPattern - 1}"))
      _ <- api.control.sequenced(deliveryIds.size, 1) { i =>
        for {
          nextPhaseId <- api.xlr.phases.appendPhase(phaseId.release)
          deliveryId = deliveryIds(i).id
          _ <- api.xlr.tasks.appendDeliveryTask(nextPhaseId, WaitForTrackedItems(s"wait all items in $deliveryId",
            deliveryId, trackedItems.toList, s"Stage-${numStagesPerPattern - 1}"))
        } yield ()
      }
      tiPhaseId <- api.xlr.releases.create(s"Master Release - Tracked Items", Some(session.user))
      _ <- api.control.sequenced(deliveryIds.size) { i =>
        for {
          _ <- api.xlr.tasks.appendDeliveryTask(tiPhaseId, WaitForTrackedItems(s"wait all items in ${deliveryIds(i).id}",
            deliveryIds(i).id, trackedItems.toList, s"Stage-${numStagesPerPattern - 1}"))
        } yield ()
      }
    } yield List(phaseId.release, tiPhaseId.release)


  def createComponentRelease(deliveryId: Delivery.ID, deliveryTitle: String, stages: Map[String, Stage.ID])(i: Int)
                            (implicit session: User.Session): IO[(Task.ID, Release.ID)] =
    for {
      initPhaseId <- api.xlr.releases.create(s"Component Release $i for $deliveryTitle", Some(session.user))
      _ <- api.xlr.tasks.appendDeliveryTask(initPhaseId, RegisterTrackedItems(s"register ti$i", deliveryId.id,
        List(s"ti$i")))
      taskId <- api.xlr.tasks.appendManual(initPhaseId, "Continue?")
      _ <- api.xlr.tasks.assignTo(taskId, session.user.username)
      _ <- api.control.sequenced(numStagesPerPattern) { j =>
        for {
          markPhaseId <- api.xlr.phases.appendPhase(initPhaseId.release)
          stageId = stages(s"s$j")
          _ <- api.xlr.tasks.appendDeliveryTask(markPhaseId, MarkTrackedItems(s"mark $j", deliveryId.id,
            List(s"ti$i"), stageId.id))
          _ <- api.xlr.tasks.appendDeliveryTask(markPhaseId, WaitForTrackedItems(s"wait $j", deliveryId.id,
            List(s"ti$i"), stageId.id))
        } yield ()
      }
    } yield taskId -> initPhaseId.release
}
