package com.xebialabs.xlrelease.stress.scenarios

import cats.Show
import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.domain.Folder._
import com.xebialabs.xlrelease.stress.domain._
import com.xebialabs.xlrelease.stress.protocol.CreateReleaseArgs
import com.xebialabs.xlrelease.stress.utils.IOHelpers._
import com.xebialabs.xlrelease.stress.utils.JsUtils
import com.xebialabs.xlrelease.stress.{API, Scenario}

object ReleaseBackground extends Scenario.ScenarioMaker[ReleaseBackground, (Int, Int, Int)]("release.background") {

  import play.api.libs.functional.syntax._
  import play.api.libs.json._

  override val defaultParameters = Json.obj(
    "numReleases" -> 8,
    "numPhases" -> 5,
    "numAutomatedTasks" -> 5
  )

  implicit val readParameters: Reads[(Int, Int, Int)] = (
    (JsPath \ "numReleases").read[Int] and
      (JsPath \ "numPhases").read[Int] and
      (JsPath \ "numAutomatedTasks").read[Int]
    ) ((a, b, c) => (a, b, c))

  def scenario(parameters: (Int, Int, Int))(implicit api: API): ReleaseBackground = parameters match {
    case (numReleases, numPhases, numAutomatedTasks) =>
      ReleaseBackground(numReleases, numPhases, numAutomatedTasks)
  }
}

case class ReleaseBackground(numReleases: Int, numPhases: Int, numAutomatedTasks: Int)
                            (implicit val api: API)
  extends Scenario {

  import spray.json._

  type Params = (Folder, Template.ID, String, List[Task.ID])

  def name: String = "ReleaseBackground"

  lazy val loginTime = api.metric.named("login")
  lazy val liveRelease = api.metric.named("live")
  lazy val archiveRelease = api.metric.named("archive")

  def setup: IO[Params] =
    api.xlr.users.admin() >>= { implicit session =>
      for {
        start <- api.control.now()
        startString = start.toString("YYYY-MM-dd_HH:mm:ss")
        folderName = s"folder-$startString"
        templateTag = s"templateTag-$startString"
        templateName = s"Template-$startString"
        _ <- api.xlr.configurations.setArchivingSettings(releaseAgeToDeleteFromJcr = 0)
        folder <- api.xlr.folders.createFolder(Folder(folderName))
        _ <- api.log.session.debug("folder created")
        phaseId <- api.xlr.templates.create(templateName, scriptUser = None, tags = List(templateTag), folderId = Some(folder.id))
        _ <- api.log.session.info(s"Created template '$templateName' in folder ${folder.show}: [${phaseId.release.id}]")
        templateId = Template.ID(phaseId.release.id)
        _ <- populatePhase(phaseId)
        _ <- api.control.sequenced(numPhases - 1) { _ =>
          appendPhase(templateId)
        }
        _ <- api.log.session.info(s"creating ${numReleases} releases, half of which will be completed and archived.")
        tasksOrReleaseToComplete <- api.control.parallel(numReleases)(
          createAndStartRelease(templateId, templateName)
        )
        releasesToComplete = tasksOrReleaseToComplete.collect {
          case Right(releaseId) => releaseId
        }
        tasksToComplete = tasksOrReleaseToComplete.collect {
          case Left(taskId) => taskId
        }
        _ <- api.log.session.info(s"Waiting for ${numReleases / 2} release to be archived... Go grab some coffee...")
        _ <- waitForReleasesToBeArchived(releasesToComplete)
      } yield (folder, templateId, templateTag, tasksToComplete)
    }

  def program(params: Params): IO[Unit] = userSequence(params._3)

  def cleanup(params: Params): IO[Unit] = params match {
    case (folder, templateId, tag, tasksToComplete) =>
      val releaseIdsToAbort = tasksToComplete.map(_.phaseId.release)
      api.xlr.users.admin() >>= { implicit session =>
        for {
          _ <- releaseIdsToAbort.map(api.xlr.releases.abort).sequence
          _ <- api.xlr.templates.delete(templateId)
          _ <- api.xlr.folders.deleteFolder(folder.id)
          _ <- api.xlr.configurations.setArchivingSettings(releaseAgeToDeleteFromJcr = 720)
        } yield ()
      }
  }

  implicit val showParams: Show[Params] = {
    case (folder: Folder, templateId: Template.ID, tag: String, tasksToComplete: List[Task.ID]) =>
      s"(folder: ${folder.show}, templateId: ${templateId.show}, tag: $tag, ${tasksToComplete.length} tasks to complete)"
  }

  private def userSequence(tag: String): IO[Unit] = {
    loginTime.measure(api.control.time(api.xlr.users.admin())) >>= { implicit session =>
      for {
        _ <- api.log.session.info("Finding template by tag")
        _ <- liveRelease.measure(api.control.time(getTemplateByTag(tag)))
        _ <- api.log.session.info("Finding live releases by tag")
        _ <- withFirstFoundReleaseBy(tag, onlyArchived = false) { releaseId =>
          api.control.repeat(3)(
            liveRelease.measure(
              api.control.time(api.xlr.releases.get(releaseId))
            )
          )
        }
        _ <- api.log.session.info("Finding archive releases by tag")
        _ <- withFirstFoundReleaseBy(tag, onlyArchived = true) { releaseId =>
          api.control.repeat(3)(
            archiveRelease.measure(
              api.control.time(api.xlr.releases.getArchived(releaseId))
            )
          )
        }
      } yield ()
    }
  }

  private def getTemplateByTag(tag: String)
                              (implicit session: User.Session): IO[JsObject] =
    for {
      templates <- api.xlr.templates.search(tag = Some(tag))()
      templateId <- JsUtils.readFirstId(templates).map(Template.ID).io
      templateJson <- api.xlr.templates.get(templateId)
    } yield templateJson

  private def withFirstFoundReleaseBy[U](tag: String, onlyArchived: Boolean = false)
                                        (f: Release.ID => IO[U])
                                        (implicit session: User.Session): IO[Unit] = {
    for {
      releases <- api.xlr.releases.search(tags = Set(tag), onlyArchived = onlyArchived)()
      _ <- releases.elements.headOption match {
        case None =>
          api.control.nop
        case Some(first) =>
          JsUtils.readIdString(first).map(Release.ID).io >>= f
      }
    } yield ()
  }

  private def createAndStartRelease(templateId: Template.ID, templateName: String)
                                   (i: Int)
                                   (implicit session: User.Session): IO[Either[Task.ID, Release.ID]] = {
    for {
      releaseId <- api.xlr.releases.createFromTemplate(templateId, CreateReleaseArgs(
        s"ReleaseFrom-${templateName}_${i}",
        variables = Map.empty,
      ))
      _ <- api.log.session.info(s"created release [${releaseId.id}], i = $i")
      taskOrReleaseToComplete <- if (i % 2 == 0) {
        api.xlr.releases.start(releaseId).map(Right.apply)
      } else {
        for {
          lastPhaseId <- api.xlr.phases.appendPhase(releaseId)
          completeMe <- api.xlr.tasks.appendManual(lastPhaseId, "Complete Me")
          _ <- api.xlr.releases.start(releaseId)
        } yield Left(completeMe)
      }
    } yield taskOrReleaseToComplete
  }

  private def waitForReleasesToBeArchived(releaseIds: List[Release.ID])
                                         (implicit session: User.Session): IO[Unit] =
    api.control.parallel(releaseIds.length) { i =>
      api.xlr.releases.waitUntilArchived(releaseIds(i))
    } >> api.control.nop

  private def populatePhase(phaseId: Phase.ID)
                           (implicit session: User.Session): IO[Unit] =
    api.control.sequenced(numAutomatedTasks) { _ =>
      appendNoopTask(phaseId)
    }.map(_ => ())

  private def appendPhase(templateId: Template.ID)
                         (implicit session: User.Session): IO[Phase.ID] =
    for {
      phaseId <- api.xlr.phases.appendPhase(Release.ID(templateId.id))
      _ <- populatePhase(phaseId)
    } yield phaseId

  private def appendNoopTask(phaseId: Phase.ID)
                            (implicit session: User.Session): IO[Task.ID] =
    api.xlr.tasks.appendScript(phaseId, "noop", "xlrelease.ScriptTask", "True")
}
