package com.xebialabs.xlrelease.stress.scenarios

import cats._
import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.domain._
import com.xebialabs.xlrelease.stress.{API, Scenario}
import play.api.libs.functional.syntax._
import play.api.libs.json._
import spray.json._

import scala.language.postfixOps

object MultipleDeliveriesScenario
  extends Scenario.ScenarioMaker[MultipleDeliveriesScenario, (Int, Int, Int, Int)]("multi.deliveries") {

  override val defaultParameters = Json.obj(
    "numDeliveriesPerPattern" -> 20,
    "numStagesPerPattern" -> 10,
    "numTrackedItemsPerPattern" -> 20,
    "concurrency" -> 8
  )

  implicit val readParameters: Reads[(Int, Int, Int, Int)] = (
    (JsPath \ "numDeliveriesPerPattern").read[Int] and
      (JsPath \ "numStagesPerPattern").read[Int] and
      (JsPath \ "numTrackedItemsPerPattern").read[Int] and
      (JsPath \ "concurrency").read[Int]
    ) ((a, b, c, d) => (a, b, c, d))

  def scenario(parameters: (Int, Int, Int, Int))(implicit api: API): MultipleDeliveriesScenario = parameters match {
    case (a, b, c, d) =>
      MultipleDeliveriesScenario(
        numDeliveriesPerPattern = a,
        numStagesPerPattern = b,
        numTrackedItemsPerPattern = c,
        concurrency = d
      )
  }
}

case class MultipleDeliveriesScenario(
                                       numDeliveriesPerPattern: Int,
                                       numStagesPerPattern: Int,
                                       numTrackedItemsPerPattern: Int,
                                       concurrency: Int)
                                     (implicit val api: API) extends Scenario with DefaultJsonProtocol {
  override type Params = (Folder, Pattern.ID)

  lazy val reachDeliveryGeneration = api.metric.named("reachDeliveryGeneration")

  def name: String = "MultipleDeliveriesScenario"

  def setup: IO[Params] =
    api.xlr.users.admin() >>= { implicit session =>
      for {
        _ <- api.log.session.debug("creating parent folder")
        start <- api.control.now()
        folderName = s"deliveries-${start.toString("YYYY-MM-dd_HH:mm:ss")}"
        folder <- api.xlr.folders.createFolder(Folder(folderName))
        stageId <- api.xlr.patterns.create(s"Pattern", folder.id.show)
        patternId = stageId.patternId
        _ <- api.control.sequenced(numStagesPerPattern) { i =>
          api.xlr.patterns.addStage(s"Stage-$i", patternId, i)
        }
        _ <- api.control.sequenced(numTrackedItemsPerPattern) { i =>
          api.xlr.patterns.addTrackedItem(s"TrackedItem-$i", patternId)
        }
      } yield (folder, patternId)
    }

  def program(params: Params): IO[Unit] =
    api.xlr.users.admin() >>= { implicit session =>
      for {
        _ <- api.log.info(s"started program for multi.deliveries with ${numDeliveriesPerPattern} deliveries per pattern")
        timedDeliveryCreation <- api.control.time(
          api.control.concurrently(concurrency) {
            (0 until numDeliveriesPerPattern).map { i =>
              reachDeliveryGeneration.measure(
                api.control.time(api.xlr.deliveries.createFromPattern(s"Delivery-${i}", params._2, params._1.id))
              )
            }
          }.map(_.flatten)
        )
        (createTime, subDeliveriesPerPattern) = timedDeliveryCreation
        _ <- api.log.info(s"created $numDeliveriesPerPattern deliveries in ${createTime.toMillis}ms")
      } yield ()
    }

  def cleanup(params: Params): IO[Unit] =
    api.log.info(s"check cleanup") >>
      api.xlr.users.admin() >>= { implicit session =>
      api.xlr.folders.deleteFolder(params._1.id)
    }

  implicit val showParams: Show[Params] = Show[Params] {
    case (folder: Folder, patternId: Pattern.ID) =>
      s"folder: ${folder.id.show}, pattern: ${patternId.show}"
  }
}
