package com.xebialabs.xlrelease.stress.scenarios

import cats.Show
import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.api.metric.MetricFunction
import com.xebialabs.xlrelease.stress.domain.Folder
import com.xebialabs.xlrelease.stress.{API, Scenario}

import scala.util.Random

object FoldersFlat extends Scenario.ScenarioMaker[FoldersFlat, (Int, Int, Int, Int)]("folders.flat") {

  import play.api.libs.functional.syntax._
  import play.api.libs.json._

  override val defaultParameters = Json.obj(
    "concurrency" -> 8,
    "numSubfolders" -> 1000,
    "numFoldersToRename" -> 250,
    "numFoldersToDelete" -> 250
  )

  implicit val readParameters: Reads[(Int, Int, Int, Int)] = (
    (JsPath \ "concurrency").read[Int] and
      (JsPath \ "numSubfolders").read[Int] and
      (JsPath \ "numFoldersToRename").read[Int] and
      (JsPath \ "numFoldersToDelete").read[Int]
    ) ((a, b, c, d) => (a, b, c, d))

  def scenario(parameters: (Int, Int, Int, Int))(implicit api: API): FoldersFlat = parameters match {
    case (a, b, c, d) => FoldersFlat(a, b, c, d)
  }
}

case class FoldersFlat(concurrency: Int, numSubfolders: Int, numFoldersToRename: Int, numFoldersToDelete: Int)
                      (implicit val api: API)
  extends Scenario {
  type Params = Folder

  require(numSubfolders > 0)
  require(numFoldersToRename <= numSubfolders)
  require(numFoldersToDelete <= numSubfolders)

  def name: String = "FoldersFlat"

  lazy val creatingFolderTime: MetricFunction = api.metric.named("creating folder time")
  lazy val renamingFolderTime: MetricFunction = api.metric.named("renaming folder time")
  lazy val deletingFolderTime: MetricFunction = api.metric.named("deleting folder time")

  // TODO: remove this, move it to Runner.scala
  def setup: IO[Params] = api.xlr.users.admin() >>= { implicit session =>
    // TODO: likewise, to Runner.scala
    for {
      _ <- api.log.session.debug("creating parent folder")
      start <- api.control.now()
      parentName = s"parent-${start.toString("YYYY-MM-dd_HH:mm:ss")}"
      parent <- api.xlr.folders.createFolder(Folder(parentName))
    } yield parent
  }

  def program(parent: Folder): IO[Unit] = {
    api.xlr.users.admin() >>= { implicit session =>
      for {
        timedSubFolders <- api.control.time(
          api.control.concurrently(concurrency) {
            (0 until numSubfolders).map { i =>
              creatingFolderTime.measure(
                api.control.time(api.xlr.folders.createFolder(Folder(s"subFolder$i", Some(parent))))
              )
            }
          }.map(_.flatten)
        )
        (createTime, subFolders) = timedSubFolders
        _ <- api.log.info(s"created $numSubfolders folders in ${createTime.toMillis}ms")

        foldersToRename = Random.shuffle(subFolders).take(numFoldersToRename)
        timedRenamedFolders <- api.control.time(
          api.control.concurrently(concurrency) {
            (0 until numFoldersToRename).map { i =>
              val folder = foldersToRename(i)
              renamingFolderTime.measure(api.control.time(
                api.xlr.folders.renameFolder(folder.id, s"${folder.name}-renamed")
              ))
            }
          }.map(_.flatten)
        )
        (renameTime, _) = timedRenamedFolders
        _ <- api.log.session.info(s"renamed $numFoldersToRename folders in ${renameTime.toMillis}ms")

        foldersToDelete = Random.shuffle(subFolders).take(numFoldersToDelete)
        timedDeletedFolders <- api.control.time(
          api.control.concurrently(concurrency) {
            (0 until numFoldersToDelete).map { i =>
              val folderId = foldersToDelete(i).id
              deletingFolderTime.measure(api.control.time(
                api.xlr.folders.deleteFolder(folderId)
              ))
            }
          }.map(_.flatten)
        )
        (deleteTime, _) = timedDeletedFolders
      } yield ()
    }
  }

  def cleanup(parent: Folder): IO[Unit] =
    api.xlr.users.admin() >>= { implicit session =>
      api.xlr.folders.deleteFolder(parent.id)
    }

  implicit val showParams: Show[Params] = Folder.showFolder
}
