package com.xebialabs.xlrelease.stress.scenarios

import cats.Show
import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.domain.Member.RoleMember
import com.xebialabs.xlrelease.stress.domain.Permission.{CreateRelease, CreateTemplate, CreateTopLevelFolder}
import com.xebialabs.xlrelease.stress.domain._
import com.xebialabs.xlrelease.stress.protocol.CreateReleaseArgs
import com.xebialabs.xlrelease.stress.utils.JsUtils.JsParsed
import com.xebialabs.xlrelease.stress.utils.{JsUtils, ScenarioUtils}
import com.xebialabs.xlrelease.stress.{API, Scenario}

import scala.concurrent.duration._
import scala.io.Source

object CompleteReleases extends Scenario.ScenarioMaker[CompleteReleases, Int]("complete_releases") {

  import play.api.libs.json._

  override val defaultParameters: JsObject = Json.obj(
    "numUsers" -> 50
  )

  implicit val readParameters: Reads[Int] = (JsPath \ "numUsers").read[Int]

  def scenario(parameters: Int)(implicit api: API): CompleteReleases =
    CompleteReleases(parameters)
}

case class CompleteReleases(numUsers: Int)
                           (implicit val api: API)
  extends Scenario
    with ScenarioUtils {

  type Params = (Role, Template.ID)

  val name: String = s"Complete $numUsers Releases "

  val masterAutomationTpl: String = Source.fromResource("MasterAutomationTemplate.groovy")
    .getLines()
    .mkString("\n")

  val masterSeedTpl: String = Source.fromResource("MasterSeedTemplate.groovy")
    .getLines()
    .mkString("\n")

  override def setup: IO[(Role, Template.ID)] =
    api.xlr.users.admin() >>= { implicit session =>
      for {
        _ <- api.log.info("setting up role....")
        role <- createUsers(numUsers) >>= createGlobalRole("superDuperRole")
        automationId <- createReleaseFromGroovy(
          "Create Master Automation Template from groovy",
          masterAutomationTpl,
          Map("###ADMIN_PASSWORD###" -> session.user.password)
        )
        seedId <- createReleaseFromGroovy(
          "Create Master Seed Template from groovy",
          masterSeedTpl,
          Map("###ADMIN_PASSWORD###" -> session.user.password)
        )
        templateId = Template.ID(seedId.id)
        _ <- setupTeams(role, Template.ID(automationId.id))
        _ <- setupTeams(role, templateId)
        _ <- api.log.info("setup role complete!")
      } yield (role, templateId)
    }

  override def program(params: (Role, Template.ID)): IO[Unit] = params match {
    case (role, templateId) =>
      val users = role.principals.toList
      val generateLoad = rampUp(1, numUsers, _ * 2) { n =>
        simple(templateId, users(n))
      }.map(_ => ())
      api.xlr.users.admin() >>= { implicit session =>
        for {
          results <- withHealthCheck(program = generateLoad, checkInterval = new FiniteDuration(1, SECONDS), checkIO = checkReleases)
          (_, health) = results
          _ <- api.log.info(s"**** RESULTS: ${health.size} entries *****")
          sorted = health.sortBy(_._2.toMillis)
          fastest = sorted.head
          slowest = sorted.last
          _ <- api.log.info(s"*** FASTEST: ${fastest._1} | ${fastest._2} | ${fastest._3.mapValues(_.size).mkString(", ")}")
          _ <- api.log.info(s"*** SLOWEST: ${slowest._1} | ${slowest._2} | ${slowest._3.mapValues(_.size).mkString(", ")}")
          _ <- health.map { case (t, d, releasesByStatus) =>
            api.log.info(s"${t.toString} | ${d.toMillis}ms | ${releasesByStatus.mapValues(_.size).mkString(", ")}")
          }.sequence
        } yield ()
      }
  }


  override def cleanup(params: (Role, Template.ID)): IO[Unit] = params match {
    case (role, _) =>
      api.xlr.users.admin() >>= { implicit session =>
        for {
          _ <- api.log.info("Cleaning up users and role")
          _ <- deleteUsers(role.principals.map(_.username).toList)
          _ <- api.xlr.users.deleteRole(role.roleName)
        } yield ()
      }
  }

  implicit val showParams: Show[(Role, Template.ID)] = {
    case (role, templateId) => s"$role, $templateId"
  }

  protected def simple(templateId: Template.ID, user: User): IO[Unit] = {
    api.xlr.users.login(user) >>= { implicit session =>
      for {
        _ <- api.log.info(s"logged in as ${user.username}...")
        _ <- api.log.info("Creating release from template")
        seedRelId <- api.xlr.releases.createFromTemplate(templateId, CreateReleaseArgs(
          title = s"Seed release by ${user.username}",
          variables = Map("user" -> user.username)
        ))
        _ <- api.log.info(s"Starting release $seedRelId")
        _ <- api.xlr.releases.start(seedRelId)
        taskIds <- api.xlr.releases.getTasksByTitle(seedRelId, "CR1")
        taskId = taskIds.head
        _ <- api.log.info(s"Waiting for task ${taskId.show}")
        _ <- api.xlr.tasks.waitFor(taskId, TaskStatus.Completed, retries = None)
        _ <- api.log.debug(s"getComments(${taskId.show})")
        comments <- api.xlr.tasks.getComments(taskId)
        _ <- api.log.debug(s"comments: ${comments.mkString("\n")}")
        comment <- comments.init.lastOption.map(api.control.ok[Comment]).getOrElse {
          api.control.fail(s"No comments in task ${taskId.show}")
        }
        _ <- api.log.debug("last comment: " + comment)
        createdRelId <- getIdFromComment(comment.text).map(id => api.control.ok(Release.ID(id.id))).getOrElse {
          api.control.fail(s"Cannot extract releaseId from comment: $comment")
        }
        manualTaskIds <- api.xlr.releases.getTasksByTitle(createdRelId, "UI")
        uiTaskId = manualTaskIds.head
        _ <- api.xlr.tasks.assignTo(uiTaskId, user.username)
        _ <- api.log.info(s"Completing task ${uiTaskId.show}")
        _ <- api.xlr.tasks.complete(uiTaskId)
        _ <- api.log.info("Waiting for release to complete")
        _ <- api.xlr.releases.waitFor(seedRelId, ReleaseStatus.Completed, retries = None)
      } yield ()
    }
  }

  private def setupTeams(role: Role, templateId: Template.ID)(implicit session: User.Session): IO[Unit] =
    for {
      _ <- api.log.info("template created: " + templateId)
      _ <- api.log.info("getting template teams...")
      teams <- api.xlr.templates.getTeams(templateId)
      teamsMap = teams.map(t => t.teamName -> t).toMap
      templateOwner = teamsMap("Template Owner") match {
        case team => team.copy(members = team.members :+ RoleMember(role.roleName))
      }
      releaseAdmin = teamsMap("Release Admin") match {
        case team => team.copy(members = team.members :+ RoleMember(role.roleName))
      }
      _ <- api.log.info("setting up teams:")
      _ <- api.log.info(templateOwner.show)
      _ <- api.log.info(releaseAdmin.show)
      _ <- api.xlr.templates.setTeams(templateId, Seq(templateOwner, releaseAdmin))
      _ <- api.log.info("template teams setup correctly")
    } yield ()

  protected def createGlobalRole(rolename: Role.ID)
                                (users: List[User])
                                (implicit session: User.Session): IO[Role] = {
    val role = Role(rolename, Set(CreateTemplate, CreateRelease, CreateTopLevelFolder), users.toSet)

    api.log.info(s"Creating global role for ${users.size} users...").flatMap { _ =>
      api.xlr.users.createRole(role).map(_ => role)
    }
  }

  protected def generateUsers(n: Int): List[User] =
    (0 until n).toList.map(i => User(s"user$i", "", "", s"user$i"))

  protected def createUsers(n: Int)
                           (implicit session: User.Session): IO[List[User]] = {
    (api.log.info(s"Creating $n users..."): IO[Unit]) >> {
      generateUsers(n).map(u =>
        api.xlr.users.createUser(u).map(_ => u)
      ).sequence: IO[List[User]]
    }
  }

  protected def deleteUsers(users: List[User.ID])
                           (implicit session: User.Session): IO[Unit] = {
    users.map(api.xlr.users.deleteUser).sequence.map(_ => ())
  }

  protected def checkReleases(implicit session: User.Session): IO[Map[ReleaseStatus, Set[Release.ID]]] =
    for {
      array <- api.xlr.releases.search(statuses = ReleaseStatus.active + ReleaseStatus.Planned)()
      raw = array.elements.toList.map(JsUtils.readReleaseIdAndStatus).sequence
    } yield {
      raw.right.get.groupBy(_._2).mapValues(_.map(_._1).toSet).toMap
    } // FIXME


}
