package com.xebialabs.xlrelease.stress.runner

import java.io.File

import cats.effect.{ContextShift, IO, Timer}
import cats.implicits._
import com.xebialabs.xlrelease.stress.config.GrafanaConfig
import com.xebialabs.xlrelease.stress.domain.User
import com.xebialabs.xlrelease.stress.utils.ZipUtils
import com.xebialabs.xlrelease.stress.{ReportingAPI, Scenario}
import org.joda.time.DateTime

import scala.concurrent.duration._
import scala.concurrent.{CancellationException, Future}
import scala.language.postfixOps

object Runner {

  def runIO(scenario: Scenario, setupTimeout: FiniteDuration, programTimeout: FiniteDuration)
           (implicit report: ReportingAPI): IO[Unit] = {
    import report.reportingXlrConfig
    import scenario.{self, showParams}

    implicit val timer: Timer[IO] = IO.timer(scenario.api.ec)
    implicit val cs: ContextShift[IO] = IO.contextShift(scenario.api.ec)

    val startTime = DateTime.now
    val reportUser = User(reportingXlrConfig.xlrConfig.username, "", "", reportingXlrConfig.xlrConfig.password)

    def error(msg: String, e: Throwable = null): IO[Unit] =
      for {
        _ <- report.log.error(msg)
        _ <- Option(e) match {
          case None => report.control.nop
          case Some(err) =>
            val stackTrace = err.getStackTrace
            if (stackTrace.nonEmpty) {
              report.log.error(stackTrace.mkString("Stack trace:\n", "\n", ""))
            } else {
              report.control.nop
            }
        }
      } yield ()

    val setup: IO[scenario.Params] = for {
      _ <- report.log.info("Clearing old admin credentials")
      _ <- scenario.api.xlr.users.clearCookies()
      _ <- report.log.info("setting up...")
      timedParams <- report.control.time(scenario.setup)
      _ <- report.log.info(s"setup complete: ${timedParams._2.show} in ${timedParams._1.toMillis}ms")
    } yield timedParams._2

    def flushMetrics(): IO[Unit] =
      for {
        _ <- report.log.info("Flushing metrics...")
        _ <- report.metricFlusher.flush(reportingXlrConfig.taskId)
        _ <- report.log.info("Metrics flushed.")
        _ <- if (reportingXlrConfig.grafanaConfig.server.enabled) {
          fetchGraphs() >>= uploadGraphs
        } else {
          report.control.nop
        }
      } yield ()

    def fetchGraphs(): IO[Seq[File]] = {
      val endTime = DateTime.now
      for {
        _ <- report.log.info("Fetching grafana graphs...")
        files <- report.control.concurrently(8) {
          implicit val dashboardsConfig: GrafanaConfig.DashboardsConfig = reportingXlrConfig.grafanaConfig.dashboards
          reportingXlrConfig.grafanaConfig.panels.toList.flatMap {
            case (dashboard, panels) =>
              panels.map(dashboard -> _)
          }.map {
            case (dashboard, panel) =>
              for {
                file <- report.grafana.fetchGraph(dashboard, panel, startTime, endTime)
                _ <- report.log.info(s"Downloaded ${dashboard.name}/${panel.name} chart: ${file.getAbsoluteFile.toString}")
              } yield file
          }
        }.map(_.flatten)
        _ <- scenario.api.log.info("Downloaded all grafana graphs, creating zip archive.")
      } yield files
    }

    def uploadGraphs(files: Seq[File]): IO[Unit] = {
      for {
        archive <- IO {
          val downloadDir = reportingXlrConfig.grafanaConfig.server.downloadDir.toAbsolutePath
          val parentDir = downloadDir.getParent
          ZipUtils.archive(
            files = files,
            dest = parentDir.resolve(s"${scenario.name}-graphs.zip").toFile,
            stripPrefix = Some(downloadDir.toString)
          )
        }
        _ <- report.xlr.users.login(reportUser) >>= { implicit session =>
          for {
            id <- report.xlr.tasks.addAttachment(reportingXlrConfig.taskId, archive)
            _ <- report.log.info(s"Uploaded attachment '$id'")
          } yield ()
        }
      } yield ()
    }

    def program(params: scenario.Params): IO[Unit] =
      for {
        _ <- report.log.info(s"Running scenario")
        time <- report.control.time(scenario.program(params)).map(_._1)
        _ <- report.log.info(s"Scenario done in ${time.toMillis}ms")
      } yield ()

    def closeTask(error: Option[Throwable]): IO[Unit] =
      report.xlr.users.login(reportUser) >>= { implicit session =>
        error.map { err =>
          for {
            _ <- report.xlr.tasks.fail(report.reportingXlrConfig.taskId, s"Scenario failed: ${err.getMessage}")
            _ <- report.log.info(s"Failed task '${reportingXlrConfig.taskId.show}'")
          } yield ()
        }.getOrElse {
          for {
            now <- report.control.now()
            _ <- report.xlr.tasks.complete(reportingXlrConfig.taskId, Some(s"Scenario completed in ${now.getMillis - startTime.getMillis}ms"))
            _ <- report.log.info(s"Task '${reportingXlrConfig.taskId.show}' completed")
          } yield ()
        }
      }

    val runSetup: IO[scenario.Params] =
      timeoutTo(setup, setupTimeout) {
        for {
          _ <- report.log.warn("Setup timed out before completion")
          error = new CancellationException("setup timed out before completion")
          _ <- closeTask(Some(error))
          e <- IO.raiseError(error)
        } yield e
      }

    def runCleanup(params: scenario.Params, error: Option[Throwable] = None): IO[Unit] =
      for {
        _ <- report.log.info(s"Cleaning up...")
        time <- report.control.time(scenario.cleanup(params)).map(_._1)
        _ <- report.log.info(s"cleanup complete in ${time.toMillis}ms")
        _ <- closeTask(error)
      } yield ()


    def runProgram(params: scenario.Params): IO[Unit] =
      (program(params) >> flushMetrics() >> runCleanup(params)).recoverWith {
        case err =>
          for {
            _ <- error("Error while executing program: " + err.toString, err)
            _ <- runCleanup(params, Some(err))
            _ <- report.control.error(err)
          } yield ()
      }

    def runProgramAndCleanup(params: scenario.Params): IO[Unit] = {
      timeoutTo(runProgram(params), programTimeout) {
        for {
          _ <- report.log.warn("Program timed out, running cleanup")
          _ <- runCleanup(params)
          _ <- report.control.fail("Program timed out")
        } yield ()
      }
    }

    def runScenario = runSetup >>= runProgramAndCleanup

    runScenario.recoverWith {
      case err => report.log.warn(s"Error while executing program: ${err.toString} ${err.getMessage}")
    } >> scenario.api.metric.clear

  }

  def runFuture(scenario: Scenario, setupTimeout: FiniteDuration, programTimeout: FiniteDuration)
               (implicit report: ReportingAPI): Future[Unit] = {
    runIO(scenario, setupTimeout, programTimeout).unsafeToFuture()
  }

  protected def timeoutTo[A](p: IO[A], after: FiniteDuration)
                            (fallback: IO[A])
                            (implicit timer: Timer[IO], cs: ContextShift[IO]): IO[A] = {
    IO.race(p, timer.sleep(after)) >>= {
      case Left(a) => a.pure[IO]
      case Right(_) => fallback
    }
  }

}
