package com.xebialabs.xlrelease.stress.runner

import com.xebialabs.xlrelease.stress.Scenario
import com.xebialabs.xlrelease.stress.domain.Task
import play.api.libs.json.{JsError, JsValue, Json, OWrites}

import scala.concurrent.Future
import scala.concurrent.duration.FiniteDuration

object Protocol {

  case class Job(scenario: String,
                 params: JsValue,
                 job: Future[Unit],
                 setupTimeout: FiniteDuration,
                 programTimeout: FiniteDuration,
                 jobStatus: JobStatus)

  sealed trait JobStatus

  object JobStatus {
    case object Running extends JobStatus
    case object Completed extends JobStatus
    case class Failed(err: Throwable) extends JobStatus

    implicit val writeJobStatus: OWrites[JobStatus] = OWrites[JobStatus] {
      case Running => Json.obj("status" -> "running")
      case Completed => Json.obj("status" -> "completed")
      case Failed(err) => Json.obj(
        "status" -> "failed",
        "error" -> err.toString
      )
    }
  }

  implicit val writeJob: OWrites[Job] = OWrites[Job](job =>
    Json.obj(
      "scenario" -> job.scenario,
      "params" -> job.params,
      "setupTimeout" -> job.setupTimeout.toSeconds,
      "programTimeout" -> job.programTimeout.toSeconds
    ) ++ Json.toJsObject(job.jobStatus)
  )

  // Requests
  case class RunScenario(scenario: String, params: JsValue, taskId: Task.ID, setupTimeout: FiniteDuration, programTimeout: FiniteDuration)
  case object GetStatus
  case object ListScenarios

  // Responses
  sealed trait RunResponse
  case class UnknownScenario(scenario: String) extends RunResponse
  case class InvalidParameters(scenario: String, error: JsError) extends RunResponse
  case class Started(job: Job) extends RunResponse
  case class Busy(job: Job) extends RunResponse
  case class ScenarioList(scenarios: List[Scenario.ScenarioMaker[_ <: Scenario, _]])

  type RunnerStatus = Option[Job]

  implicit val writeRunResponse: OWrites[RunResponse] = OWrites[RunResponse] {
    case Started(job) =>
      Json.toJsObject(job) + ("status" -> Json.toJson("started"))

    case Busy(job) =>
      Json.toJsObject(job)

    case UnknownScenario(scenario) =>
      Json.obj(
        "error" -> "unknown scenario",
        "scenario" -> scenario
      )

    case InvalidParameters(scenario, error) =>
      Json.obj(
        "error" -> "invalid parameters",
        "scenario" -> scenario,
        "message" -> JsError.toJson(error)
      )
  }

  implicit val writeScenarioList: OWrites[ScenarioList] = OWrites[ScenarioList] {
    case ScenarioList(scenarios) =>
      Json.toJsObject(
        scenarios.map(scenario => scenario.name.toLowerCase -> scenario.defaultParameters).toMap
      )
  }

  implicit val writeGetStatusResponse: OWrites[RunnerStatus] = OWrites[RunnerStatus] {
    case None => Json.obj("status" -> "idle")
    case Some(job) => Json.toJsObject(job)
  }

}
