package com.xebialabs.xlrelease.stress.runner

import java.nio.file.Paths
import java.util.concurrent.{ExecutorService, Executors}

import akka.actor.{Actor, Props}
import akka.http.scaladsl.model.Uri
import com.xebialabs.xlrelease.stress.Scenario.ScenarioMaker
import com.xebialabs.xlrelease.stress.config.GrafanaConfig.{DashboardConfig, PanelConfig}
import com.xebialabs.xlrelease.stress.config._
import com.xebialabs.xlrelease.stress.domain.{Phase, Release, Task}
import com.xebialabs.xlrelease.stress.runner.Protocol._
import com.xebialabs.xlrelease.stress.scenarios._
import com.xebialabs.xlrelease.stress.scenarios.meta.ScenarioWithBackground.MakeScenarioWithBackground
import com.xebialabs.xlrelease.stress.{API, Scenario}
import play.api.libs.json._
import play.api.{Configuration, Logging}

import scala.concurrent.ExecutionContext
import scala.concurrent.duration.FiniteDuration
import scala.util.{Failure, Success}

object ActorRunner {
  def props(config: Configuration): Props = Props(new ActorRunner(config))
}

class ActorRunner(config: Configuration) extends Actor with Logging {

  private val running: RunnerStatus = Option.empty

  // TODO: extract config parsing
  lazy val xlrConfig: XlrConfig = {
    val targetUrl = config.get[String]("server.url")
    val server = XlrServer(Uri(targetUrl))
    val adminPassword = config.get[String]("server.adminPassword")
    XlrConfig(server, "admin", adminPassword)
  }
  implicit lazy val api: API = new API(xlrConfig)

  val reportConfig: Task.ID => ReportingXlrConfig = taskId => {
    val dashboardsProperties = Set("orgId", "width", "height", "timezone", "download-dir")
    val panelProperties = Set("id", "name", "panels")
    ReportingXlrConfig(
      xlrConfig = XlrConfig(
        server = XlrServer(Uri(config.get[String]("reportServer.url"))),
        username = config.get[String]("reportServer.username"),
        password = config.get[String]("reportServer.password")
      ),
      taskId = taskId,
      esConfig = ElasticsearchConfig(
        url = config.get[String]("elasticsearch.url"),
        index = config.get[String]("elasticsearch.index"),
        enabled = config.get[Boolean]("elasticsearch.enabled")
      ),
      grafanaConfig = GrafanaConfig(
        server = GrafanaConfig.ServerConfig(
          url = config.get[String]("grafana.url"),
          token = config.get[String]("grafana.token"),
          enabled = config.get[Boolean]("grafana.enabled"),
            downloadDir = Paths.get(config.get[String]("grafana.download-dir"))
        ),
        dashboards = GrafanaConfig.DashboardsConfig(
          orgId = config.get[Int]("grafana.dashboards.orgId"),
          width = config.get[Int]("grafana.dashboards.width"),
          height = config.get[Int]("grafana.dashboards.height"),
          timezone = config.get[String]("grafana.dashboards.timezone")
        ),
        panels = {
          config.get[Configuration]("grafana.dashboards").subKeys.diff(dashboardsProperties).map { dashboardName =>
            logger.info(s"loading dashboard config: grafana.dashboards.$dashboardName")
            val dashboardConfig = config.get[Configuration](s"grafana.dashboards.$dashboardName")

            val dashboard = DashboardConfig(
              id = dashboardConfig.get[String]("id"),
              name = dashboardConfig.get[String]("name"),
              parameters = dashboardConfig.subKeys.diff(panelProperties).map(parameterName =>
                parameterName -> dashboardConfig.get[String](parameterName)
              ).toMap
            )
            val panels = dashboardConfig.get[Configuration]("panels").keys.map { panelName =>
              PanelConfig(id = dashboardConfig.get[Int](s"panels.$panelName"), name = panelName)
            }
            dashboard -> panels
          }.toMap
        }
      )
    )
  }

  lazy val scenarioPool: ExecutorService = Executors.newFixedThreadPool(config.get[Int]("client.numThreads"))
  lazy val scenarioEC: ExecutionContext = ExecutionContext.fromExecutor(scenarioPool)

  lazy val runnerExecutor: ExecutorService = Executors.newSingleThreadExecutor()
  implicit lazy val runnerEC: ExecutionContext = ExecutionContext.fromExecutor(runnerExecutor)

  lazy val scenarios: Map[String, Scenario.ScenarioMaker[_ <: Scenario, _]] = Seq[Scenario.ScenarioMaker[_ <: Scenario, _]](
    CommentsScenario,
    Generate,
    FoldersFlat,
    SubReleases,
    MultipleDeliveriesScenario,
    ReleasesPerDeliveryScenario,
    WebhooksRequestsBaselineScenario
  ).map(sm => sm.name.toLowerCase -> sm).toMap

  logger.debug("Registered scenarios:")
  scenarios.keySet.foreach(name => logger.debug(s"'$name'"))

  logger.trace(s"API configuration: ${xlrConfig}")
  logger.trace(s"ReportAPI configuration: ${reportConfig(Task.ID(Phase.ID(Release.ID("FakeRelease"), "FakePhase"), "FakeTask"))}")

  def receive: Receive = onMessage(running)

  private def getScenario(scenario: String): Option[ScenarioMaker[_ <: Scenario, _]] =
    scenarios.get(scenario.toLowerCase).orElse {
      parseCompositeScenario(scenario).collect {
        case Right(mkScenario) => mkScenario
      }
    }

  private def runScenario(mkScenario: ScenarioMaker[_ <: Scenario, _],
                          params: JsValue,
                          taskId: Task.ID,
                          setupTimeout: FiniteDuration,
                          programTimeout: FiniteDuration,
                          scenario: String): RunResponse = {
    running match {
      case Some(job) if !job.job.isCompleted =>
        Busy(job)
      case _ =>
        implicit val reportingXlrConfig: ReportingXlrConfig = reportConfig.apply(taskId)
        mkScenario.runJsonFuture(params, setupTimeout, programTimeout) match {
          case err: JsError =>
            InvalidParameters(scenario, err)
          case JsSuccess(runningFuture, _) =>
            runningFuture.onComplete {
              case Success(_) =>
                context.become(onMessage(running.map(_.copy(jobStatus = JobStatus.Completed))))
              case Failure(err) =>
                logger.error(s"Failure while running scenario '$scenario':", err)
                context.become(onMessage(running.map(job => job.copy(jobStatus = JobStatus.Failed(err)))))
                logger.info("running is now: " + Json.prettyPrint(Json.toJsObject(running)))
            }
            val job = Job(scenario, params, runningFuture, setupTimeout, programTimeout, JobStatus.Running)
            context.become(onMessage(Some(job)))
            Started(job)
        }
    }
  }

  // TODO: support nesting.
  def parseCompositeScenario(name: String): Option[Either[String, _ <: MakeScenarioWithBackground[_ <: Scenario, _, _ <: Scenario, _]]] = {
    val lowerCase = name.toLowerCase
    lowerCase match {
      case _ if lowerCase contains "_withbackground_" =>
        Some {
          name.toLowerCase.split("_withbackground_").toList match {
            case fgName :: bgName :: Nil =>
              for {
                fg <- scenarios.get(fgName).toRight(s"Unknown foreground scenario: '$fgName'")
                bg <- scenarios.get(bgName).toRight(s"Unknown background scenario: '$bgName'")
              } yield MakeScenarioWithBackground(fg, bg)
            case _ =>
              Left(s"Cannot parse withBackground composite scenario: '$name'")
          }
        }
      case _ =>
        None
    }
  }

  private def onMessage(running: Protocol.RunnerStatus): Receive = {
    case GetStatus =>
      logger.trace("> GetStatus")
      logger.trace(s"< ${Json.prettyPrint(Json.toJsObject(running))}")
      sender() ! running

    case RunScenario(scenario, params, taskId, setupTimeout, programTimeout) =>
      logger.trace(s"> RunScenario($scenario, $params, $setupTimeout, $programTimeout)")
      val resp = getScenario(scenario.toLowerCase).fold[RunResponse](UnknownScenario(scenario))(
        runScenario(_, params, taskId, setupTimeout, programTimeout, scenario)
      )
      logger.trace(s"< ${Json.prettyPrint(Json.toJsObject(resp))}")
      sender() ! resp

    case ListScenarios =>
      sender() ! ScenarioList(scenarios.values.toList)
  }
}
