package com.xebialabs.xlrelease.stress.handlers

import java.time.Instant
import java.util.Date

import akka.http.scaladsl.model.Uri
import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.api.metric.MetricFunction
import com.xebialabs.xlrelease.stress.api.xlr.XlrAPI
import com.xebialabs.xlrelease.stress.config.ReportingXlrConfig
import com.xebialabs.xlrelease.stress.domain.{Task, User}
import com.xebialabs.xlrelease.stress.handlers.MetricHandler.namedMeterWriter
import com.xebialabs.xlrelease.stress.utils.HttpHelpers._
import com.xebialabs.xlrelease.stress.{Scenario, api}
import spray.json.{JsObject, _}

import scala.collection.mutable
import scala.concurrent.duration.FiniteDuration

class Meter(var min: Long, var max: Long, var count: Long, var total: Long) {
  def +=(millis: Long): Unit = this.synchronized {
    if (min > millis || count == 0) {
      min = millis
    }
    if (max < millis || count == 0) {
      max = millis
    }
    count += 1
    total += millis
  }
}

object Meter extends DefaultJsonProtocol {
  def zero = new Meter(0, 0, 0, 0)

  def newMetricFunction(meter: Meter): MetricFunction = (duration: FiniteDuration) => {
    meter += duration.toMillis
    ().pure[IO]
  }

  implicit class MeterOps(val _meter: Meter) extends AnyVal {
    def metricFunction: MetricFunction = newMetricFunction(_meter)
  }

}

object MetricHandler extends DefaultJsonProtocol {
  implicit def namedMeterWriter(implicit ts: Instant): JsonWriter[(String, Meter)] = {
    case (name: String, meter: Meter) if meter.count <= 0 =>
      JsObject(
        "metric" -> name.toJson,
        "ts" -> ts.toEpochMilli.toJson,
      )
    case (name: String, meter: Meter) =>
      JsObject(
        "metric" -> name.toJson,
        "ts" -> new Date().toInstant.toEpochMilli.toJson,
        "min" -> meter.min.toJson,
        "max" -> meter.max.toJson,
        "avg" -> (meter.total / meter.count).toJson
      )
  }
}

class MetricHandler()(implicit
                      http: api.http.Client,
                      log: api.log.Raw,
                      control: api.control.Control with api.control.Flow
) extends api.metric.Metric with DefaultJsonProtocol {

  private val state: mutable.Map[String, Meter] = mutable.Map.empty

  override def named(name: String)(implicit scenario: Scenario): MetricFunction = {
    val fullName = s"${scenario.fullName}: $name"

    val data = state.synchronized {
      state.getOrElseUpdate(fullName, Meter.zero)
    }
    data.metricFunction
  }

  override def clear(): IO[Unit] = IO(state.clear())

  override def getState: IO[Map[String, Meter]] = IO(state.toMap)
}

class MetricsFlusherHandler(config: ReportingXlrConfig)
                           (implicit
                     http: api.http.Client,
                     metric: api.metric.Metric,
                     log: api.log.Logging with api.log.Session,
                     xlr: XlrAPI
                    ) extends api.metric.MetricFlusher with DefaultJsonProtocol {

  import config.{esConfig, xlrConfig}

  lazy val user = User(xlrConfig.username, "", "", xlrConfig.password)

  def flush(taskId: Task.ID)
           (implicit scenario: Scenario): IO[Unit] = {
    implicit val ts: Instant = new Date().toInstant
    for {
      _ <- flushToES()
      _ <- flushToXLR(taskId)
      _ <- metric.clear()
    } yield ()
  }

  private def flushToES()
                       (implicit ts: Instant, scenario: Scenario): IO[Unit] = {
    if (config.esConfig.enabled) {
      http.put(
        s"${esConfig.url}/${esConfig.index}",
        JsObject(
          "mappings" -> JsObject(
            esConfig.index -> JsObject(
              "properties" -> JsObject(
                "ts" -> JsObject("type" -> "date".toJson),
                "metric" -> JsObject("type" -> "keyword".toJson),
                "min" -> JsObject("type" -> "long".toJson),
                "max" -> JsObject("type" -> "long".toJson),
                "avg" -> JsObject("type" -> "long".toJson)
              )
            )
          )
        ).toHttpEntity
      ).flatMap(http.discard).flatMap(_ => {
        metric.getState >>= (_.toList.map(postNamedMetric).sequence)
      })
    } else {
      metric.getState >>= (_.toList.map(logNamedMetric).sequence)
    }
  }.map(_ => ())

  private def flushToXLR(taskId: Task.ID)
                        (implicit ts: Instant, scenario: Scenario): IO[Unit] = {
    xlr.users.login(user) >>= { implicit session =>
      metric.getState >>= (_.toList.map { meter =>
        xlr.tasks.comment(taskId, meter.toJson.prettyPrint.replace("\n", "\n\n"))
      }.sequence.map(_ => ()))
    }
  }

  private def postNamedMetric(namedMeter: (String, Meter))
                             (implicit ts: Instant, scenario: Scenario): IO[Unit] = {
    val uri: Uri = Uri(esConfig.url).withPath(Uri.Path.Empty / esConfig.index / esConfig.index)
    log.info(s"Pushing metric to ${uri.toString()}: '${namedMeter.toJson.prettyPrint}' ") >>
      http.post(uri, namedMeter.toJson.toHttpEntity) >>=
      http.discard
  }

  private def logNamedMetric(namedMeter: (String, Meter))
                            (implicit ts: Instant, scenario: Scenario): IO[Unit] =
    log.info(namedMeter.toJson.prettyPrint)

}