package com.xebialabs.xlrelease.stress.domain

import akka.http.scaladsl.model.Uri
import cats.Show
import cats.implicits._
import com.xebialabs.xlrelease.stress.utils.HttpHelpers.IdAsUriPath
import spray.json._

case class Variable[T](id: Option[Variable.ID],
                       key: String,
                       variableType: String,
                       value: Option[T],
                       required: Boolean = false,
                       showOnReleaseStart: Boolean = false)
                      (implicit val writer: JsonWriter[T])

object Variable extends DefaultJsonProtocol {

  case class ID(release: Option[Release.ID], id: String)

  object ID {
    implicit val showVariableID: Show[ID] = {
      case ID(None, id) => id
      case ID(Some(releaseId), id) => s"${releaseId.show}/$id"
    }

    val empty = ID(None, "")
  }

  implicit def showVariable[A](implicit showValue: Show[A]): Show[Variable[A]] = Show[Variable[A]] {
    case Variable(id, key, variableType, value: Option[A], required, showOnReleaseStart) =>
      "${" + key + "} := " + value.map(showValue.show).orNull + s" [${variableType}, required=$required, showOnReleaseStart=$showOnReleaseStart]"
  }

  def string(id: Option[Variable.ID],
             key: String,
             value: Option[String],
             required: Boolean = false,
             showOnReleaseStart: Boolean = false): Variable[String] =
    Variable(id, key, "xlrelease.StringVariable", value, required, showOnReleaseStart)

  def mapStringString(id: Option[Variable.ID],
                      key: String,
                      value: Option[Map[String, String]],
                      required: Boolean = false,
                      showOnReleaseStart: Boolean = false): Variable[Map[String, String]] =
    Variable(id, key, "xlrelease.MapStringStringVariable", value, required, showOnReleaseStart)

  def listString(id: Option[Variable.ID],
                      key: String,
                      value: Option[List[String]],
                      required: Boolean = false,
                      showOnReleaseStart: Boolean = false): Variable[List[String]] =
    Variable(id, key, "xlrelease.ListStringVariable", value, required, showOnReleaseStart)

  implicit def jsonWriter[T]: JsonWriter[Variable[T]] = {
    v =>
      JsObject(
        "id" -> v.id.map(_.show).toJson,
        "key" -> v.key.toJson,
        "type" -> v.variableType.toJson,
        "value" -> v.value.map(v.writer.write).toJson,
        "requiresValue" -> v.required.toJson,
        "showOnReleaseStart" -> v.showOnReleaseStart.toJson
      )
  }

  implicit class VariableIDOps(val id: Variable.ID) extends AnyVal {
    def path: Uri.Path = id.show.asPath
  }

}
