package com.xebialabs.xlrelease.stress.domain

import akka.http.scaladsl.model.Uri
import cats.Show
import cats.implicits._
import com.xebialabs.xlrelease.stress.utils.HttpHelpers.IdAsUriPath
import spray.json._

import scala.language.implicitConversions

case class Task(id: Task.ID, title: String, taskType: String, hasDependencies: Boolean = false) {
  def isManual: Boolean =
    taskType == "xlrelease.Task" ||
      taskType == "xlrelease.GateTask" && !hasDependencies ||
      taskType == "xlrelease.UserInputTask"
}

object Task {
  case class ID(phaseId: Phase.ID, task: String)

  object ID {
    def apply(rawId: String): Task.ID = {
      val parts = rawId.replace("Applications/", "").split("/").toList
      val releaseParts = parts.takeWhile(part => !part.startsWith("Phase"))
      val releaseId = Release.ID(releaseParts.mkString("/"))
      val phaseId = Phase.ID(releaseId, parts.drop(releaseParts.length).head)
      new Task.ID(phaseId, parts.drop(releaseParts.length + 1).mkString("/"))
    }
  }

  implicit def showTaskId(implicit sp: Show[Phase.ID]): Show[Task.ID] = {
    case ID(phaseId, taskId) => s"${sp.show(phaseId)}/$taskId"
  }

  implicit class TaskIDOps(val id: Task.ID) extends AnyVal {
    def release: Release.ID = id.phaseId.release
    def path: Uri.Path = id.show.asPath
  }

  object DeliveryTask extends DefaultJsonProtocol {
    implicit val jsonWriter: JsonWriter[DeliveryTask] = new JsonWriter[DeliveryTask] {
      def common(task: DeliveryTask): Map[String, JsValue] = Map(
        "id" -> JsNull,
        "type" -> task.taskType.toJson,
        "deliveryId" -> task.deliveryId.toJson
      )

      override def write(obj: DeliveryTask): JsValue = JsObject(
        common(obj) ++ {
          obj match {
            case r: RegisterTrackedItems => Map("trackedItems" -> r.trackedItems.toJson)
            case w: WaitForTrackedItems => Map("trackedItems" -> w.trackedItems.toJson, "stage" -> w.stage.toJson)
            case m: MarkTrackedItems => Map("trackedItems" -> m.trackedItems.toJson, "stage" -> m.stage.toJson,
              "precedingStages" -> m.precedingStages.toJson)
            case ws: WaitForStage => Map("stage" -> ws.stage.toJson)
          }
        }
      )
    }
  }

  sealed abstract class DeliveryTask(val taskType: String) {
    def title: String
    def deliveryId: String
  }
  case class MarkTrackedItems(title: String, deliveryId: String, trackedItems: List[String], stage: String,
                              precedingStages: Boolean = false) extends DeliveryTask("delivery.MarkTrackedItems")
  case class RegisterTrackedItems(title: String, deliveryId: String, trackedItems: List[String])
    extends DeliveryTask("delivery.RegisterTrackedItems")
  case class WaitForTrackedItems(title: String, deliveryId: String, trackedItems: List[String], stage: String)
    extends DeliveryTask("delivery.WaitForTrackedItems")
  case class WaitForStage(title: String, deliveryId: String, stage: String) extends DeliveryTask("delivery.WaitForStage")
}