package com.xebialabs.xlrelease.stress.domain

import cats.Show
import spray.json._

sealed abstract class ReleaseStatus(val name: String)

object ReleaseStatus extends DefaultJsonProtocol {
  case object Template extends ReleaseStatus("template")
  case object Planned extends ReleaseStatus("planned")
  case object InProgress extends ReleaseStatus("inProgress")
  case object Paused extends ReleaseStatus("paused")
  case object Completed extends ReleaseStatus("completed")
  case object Failing extends ReleaseStatus("failing")
  case object Failed extends ReleaseStatus("failed")
  case object Aborted extends ReleaseStatus("aborted")

  val active: Set[ReleaseStatus] = Set(
    InProgress, Paused, Failing, Failed
  )

  val inactive: Set[ReleaseStatus] = Set(
    Completed, Aborted
  )

  implicit val showReleaseStatus: Show[ReleaseStatus] = {
    case Template => "TEMPLATE"
    case Planned => "PLANNED"
    case InProgress => "IN_PROGRESS"
    case Paused => "PAUSED"
    case Completed => "COMPLETED"
    case Failing => "FAILING"
    case Failed => "FAILED"
    case Aborted => "ABORTED"
  }

  val fromString: PartialFunction[String, ReleaseStatus] = {
    case "TEMPLATE" => Template
    case "PLANNED" => Planned
    case "IN_PROGRESS" => InProgress
    case "PAUSED" => Paused
    case "COMPLETED" => Completed
    case "FAILING" => Failing
    case "FAILED" => Failed
    case "ABORTED" => Aborted
  }

  implicit val releaseStatusReader: RootJsonReader[ReleaseStatus] = {
    case JsString(str) => fromString.lift(str).getOrElse {
      deserializationError(s"Unknown release status: $str")
    }
    case _ => deserializationError("Wrong type for release status, expected JsString")
  }
  implicit val releaseStatusWriter: RootJsonWriter[ReleaseStatus] = ts => showReleaseStatus.show(ts).toJson

  implicit val releaseStatusFormat: RootJsonFormat[ReleaseStatus] = rootJsonFormat(releaseStatusReader, releaseStatusWriter)

}