package com.xebialabs.xlrelease.stress.api.xlr

import akka.http.scaladsl.model.headers.{Cookie, HttpCookiePair, `Set-Cookie`}
import cats.data.NonEmptyList
import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.config.XlrServer
import com.xebialabs.xlrelease.stress.domain.{HttpSession, Role, User}
import com.xebialabs.xlrelease.stress.utils.HttpHelpers.JsonToHttpEntity
import com.xebialabs.xlrelease.stress.utils.IOHelpers.EitherToIO
import com.xebialabs.xlrelease.stress.utils.JsUtils
import com.xebialabs.xlrelease.stress.{Scenario, api}
import spray.json._


class Users(server: XlrServer, username: String, password: String)
           (implicit
            http: api.http.Client with api.http.Session,
            control: api.control.Control,
            json: api.json.JsonParser,
            log: api.log.Logging with api.log.Session) extends DefaultJsonProtocol { self =>

  protected var _adminSession: Option[HttpSession] = None

  protected lazy val adminUser: User = User(username, "", "", password)

  def clearCookies()(implicit scenario: Scenario): IO[Unit] = IO.pure {
    _adminSession = None
  }

  protected def adminLogin()(implicit scenario: Scenario): IO[HttpSession] =
    login(adminUser).map { session =>
      _adminSession = Some(session)
      session
    }

  def login(user: User)(implicit scenario: Scenario): IO[HttpSession] =
    for {
      _ <- log.debug(s"xlr.users.login(${user.username})")
      resp <- http.post(server.root(_ ?/ "login"),
        JsObject(
          "username" -> user.username.toJson,
          "password" -> user.password.toJson
        ).toHttpEntity)
      cookies <- getCookies(resp.headers[`Set-Cookie`])
      _ <- http.discard(resp)
      session = HttpSession(user, Cookie(cookies.toList.map(c => HttpCookiePair(c.cookie.name, c.cookie.value))))
    } yield session

  def admin()(implicit scenario: Scenario): IO[HttpSession] =
    _adminSession.fold(adminLogin())(_.pure[IO])

  def createUser(user: User)(implicit session: User.Session, scenario: Scenario): IO[User.ID] =
    for {
      _ <- log.session.debug(s"xlr.users.createUser(${user.username})")
      resp <- http.session.post(server.api(_ ?/ "users" / user.username),
        JsObject(
          "fullName" -> user.fullName.toJson,
          "email" -> user.email.toJson,
          "loginAllowed" -> true.toJson,
          "password" -> user.password.toJson
        ).toHttpEntity)
      content <- json.parse(resp)
      userId <- JsUtils.readUsername(content).io
    } yield userId

  def createRole(role: Role)(implicit session: User.Session, scenario: Scenario): IO[Role.ID] =
    for {
      _ <- log.session.debug(s"xlr.users.createRole(${role.roleName})")
      resp <- http.session.post(server.api(_ ?/ "roles" / role.roleName),
        JsObject(
          "name" -> role.roleName.toJson,
          "permissions" -> role.permissions.map(_.permission.toJson).toJson,
          "principals" -> role.principals.map(user => JsObject("username" -> user.username.toJson)).toJson
        ).toHttpEntity)
      _ <- http.discard(resp)
    } yield role.roleName

  def deleteUser(userId: User.ID)(implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.users.deleteUser($userId)")
      resp <- http.session.delete(server.api(_ ?/ "users" / userId))
      _ <- http.discard(resp)
    } yield ()

  def deleteRole(roleId: Role.ID)(implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.users.deleteRole($roleId)")
      resp <- http.session.delete(server.api(_ ?/ "roles" / roleId))
      _ <- http.discard(resp)
    } yield ()

  protected def getCookies(headers: Seq[`Set-Cookie`]): IO[NonEmptyList[`Set-Cookie`]] =
    headers.toList.toNel.map(control.ok).getOrElse {
      control.fail("lib.xlr.users: No login cookies for you!")
    }

}
