package com.xebialabs.xlrelease.stress.api.xlr

import akka.http.scaladsl.model.Uri
import cats.effect.IO
import cats.implicits._
import com.github.nscala_time.time.Imports.DateTime
import com.xebialabs.xlrelease.stress.{Scenario, api}
import com.xebialabs.xlrelease.stress.config.XlrServer
import com.xebialabs.xlrelease.stress.domain._
import com.xebialabs.xlrelease.stress.protocol.DateFormat
import com.xebialabs.xlrelease.stress.utils.JsUtils
import com.xebialabs.xlrelease.stress.utils.HttpHelpers.{JsonToHttpEntity, MultipartZip}
import com.xebialabs.xlrelease.stress.utils.IOHelpers.EitherToIO
import spray.json._


class Templates(server: XlrServer)
               (implicit
                http: api.http.Client with api.http.Session,
                control: api.control.Control with api.control.Flow,
                json: api.json.JsonParser,
                log: api.log.Logging with api.log.Session) extends DefaultJsonProtocol with DateFormat {

  def create(title: String,
             scriptUser: Option[User] = None,
             tags: List[String] = List.empty,
             folderId: Option[Folder.ID] = None)
            (implicit session: User.Session, scenario: Scenario): IO[Phase.ID] = {
    val query = Uri.Query(Map(
      "folderId" -> folderId.map("Applications/" + _.id)
    ).collect {
      case (k, Some(v)) => k -> v
    })
    val payload = JsObject(
      "id" -> JsNull,
      "type" -> "xlrelease.Release".toJson,
      "status" -> "TEMPLATE".toJson,
      "title" -> title.toJson,
      "scheduledStartDate" -> DateTime.now.toString.toJson,
      "scriptUsername" -> scriptUser.map(_.username.toJson).getOrElse(JsNull),
      "scriptUserPassword" -> scriptUser.map(_.password.toJson).getOrElse(JsNull),
      "tags" -> tags.toJson
    )
    for {
      _ <- log.session.debug(s"xlr.templates.create($title)")
      resp <- http.session.post(server.api(_ ?/ "templates").withQuery(query), payload.toHttpEntity)
      content <- json.parse(resp)
      phaseId <- JsUtils.readFirstPhaseId(sep = "/")(content).io
    } yield phaseId
  }

  def importXlr(template: Template)
               (implicit session: User.Session, scenario: Scenario): IO[Template.ID] =
    for {
      _ <- log.session.debug(s"xlr.templates.importXlr(${template.name})")
      resp <- http.session.post(server.api(_ ?/ "templates" / "import"), MultipartZip(template.xlrTemplate).toEntity)
      content <- json.parse(resp)
      templateId <- JsUtils.readFirstId(content).io
    } yield Template.ID(templateId)

  def get(templateId: Template.ID)
         (implicit session: User.Session, scenario: Scenario): IO[JsObject] =
    for {
      _ <- log.session.debug(s"xlr.templates.getTemplate(${templateId.show})")
      resp <- http.session.get(server.api(_ ?/ "templates" / "Applications" ++ templateId.path))
      templateJson <- json.parse(resp)
      templateJsonObj <- JsUtils.jsObject(templateJson).io
    } yield templateJsonObj

  def delete(templateId: Template.ID)
            (implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.templates.deleteTemplate(${templateId.show})")
      resp <- http.session.delete(server.api(_ ?/ "templates" / "Applications" ++ templateId.path))
      _ <- http.discard(resp)
    } yield ()

  def getTeams(templateId: Template.ID)
              (implicit session: User.Session, scenario: Scenario): IO[Seq[Team]] =
    for {
      _ <- log.session.debug(s"xlr.templates.getTeams(${templateId.show})")
      resp <- http.session.get(server.api(_ ?/ "templates" / "Applications" ++ templateId.path / "teams"))
      content <- json.parse(resp)
      teams <- JsUtils.readTeams(content).io
    } yield teams

  def setTeams(templateId: Template.ID, teams: Seq[Team])
              (implicit session: User.Session, scenario: Scenario): IO[Map[String, String]] =
    for {
      _ <- log.session.debug(s"xlr.templates.setTeams(${templateId.show}, ${teams.map(_.teamName).mkString("[", ", ", "]")})")
      resp <- http.session.post(server.api(_ ?/ "templates" / "Applications" ++ templateId.path / "teams"), teams.map(_.toJson).toJson.toHttpEntity)
      content <- json.parse(resp)
      teamIds <- JsUtils.readTeamIds(content).io
    } yield teamIds

  def setScriptUser(templateId: Template.ID, scriptUser: Option[User])
                   (implicit session: User.Session, scenario: Scenario): IO[Unit] = {
    val user = scriptUser.getOrElse(session.user)
    for {
      _ <- log.session.debug(s"xlr.templates.setScriptUser(${templateId.show}, $scriptUser)")
      resp <- http.session.put(server.api(_ ?/ "templates" / "Applications" ++ templateId.path),
        JsObject(
          "id" -> JsNull,
          "scheduledStartDate" -> DateTime.now.toJson,
          "type" -> "xlrelease.Release".toJson,
          "scriptUsername" -> user.username.toJson,
          "scriptUserPassword" -> user.password.toJson
        ).toHttpEntity
      )
      _ <- http.discard(resp)
    } yield ()
  }

  def search(title: Option[String] = None, tag: Option[String] = None)
            (page: Int = 0, resultsPerPage: Int = 100)
            (implicit session: User.Session, scenario: Scenario): IO[JsArray] =
    for {
      _ <- log.session.debug(s"xlr.templates.search($title, $tag)($page, $resultsPerPage)")
      params = Map(
        "page" -> Some(page.toString),
        "resultsPerPage" -> Some(resultsPerPage.toString),
        "title" -> title,
        "tag" -> tag
      )
      query = Uri.Query(params.collect { case (k, Some(v)) => k -> v })
      resp <- http.session.get(server.api(_ ?/ "templates").withQuery(query))
      content <- json.parse(resp)
      templates <- JsUtils.jsArray(content).io
    } yield templates
}