package com.xebialabs.xlrelease.stress.api.xlr

import java.io.File

import akka.http.scaladsl.model.Uri
import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.config.XlrServer
import com.xebialabs.xlrelease.stress.domain.Target._
import com.xebialabs.xlrelease.stress.domain.Task.DeliveryTask
import com.xebialabs.xlrelease.stress.domain._
import com.xebialabs.xlrelease.stress.utils.HttpHelpers._
import com.xebialabs.xlrelease.stress.utils.IOHelpers.EitherToIO
import com.xebialabs.xlrelease.stress.utils.JsUtils
import com.xebialabs.xlrelease.stress.{Scenario, api}
import spray.json._

import scala.concurrent.duration._
import scala.language.postfixOps

class Tasks(server: XlrServer)
           (implicit
            http: api.http.Client with api.http.Session,
            json: api.json.JsonParser,
            phases: api.xlr.Phases,
            control: api.control.Control with api.control.Flow,
            log: api.log.Logging with api.log.Session) extends DefaultJsonProtocol {

  def append(task: JsObject, container: ConcreteTarget)
            (implicit session: User.Session, scenario: Scenario): IO[Task.ID] =
    for {
      _ <- log.session.debug(s"xlr.tasks.append(${task.compactPrint}, ${container.show})")
      resp <- http.session.post(server.api(_ ?/ "tasks" / "Applications" ++ container.path / "tasks"), task.toHttpEntity)
      content <- json.parse(resp)
      taskId <- JsUtils.readTaskId(sep = "/")(content).io
    } yield taskId

  def get(taskId: Task.ID)
         (implicit session: User.Session, scenario: Scenario): IO[JsObject] =
    for {
      _ <- log.session.debug(s"xlr.tasks.get(${taskId.show})")
      resp <- http.session.get(server.api(_ ?/ "tasks" / "Applications" ++ taskId.path))
      content <- json.parse(resp)
      task <- JsUtils.jsObject(content).io
    } yield task

  def delete(taskId: Task.ID)
            (implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.tasks.delete(${taskId.show})")
      resp <- http.session.delete(server.api(_ ?/ "tasks" / "Applications" ++ taskId.path))
      _ <- http.discard(resp)
    } yield ()

  def copy(taskId: Task.ID, phaseId: Phase.ID, position: Int = 0)
          (implicit session: User.Session, scenario: Scenario): IO[Task.ID] =
    for {
      _ <- log.session.debug(s"xlr.tasks.copy(${taskId.show}, ${phaseId.show})")
      resp <- http.session.post(
        server.api(_ ?/ "tasks" / "Applications" ++ taskId.path / "copy").withQuery(Uri.Query(
          "targetContainerId" -> s"Applications/${phaseId.show}",
          "targetPosition" -> position.toString
        )),
        JsNull.toHttpEntity
      )
      content <- json.parse(resp)
      newTaskId <- JsUtils.readTaskId(sep = "/")(content).io
    } yield newTaskId

  def move(taskId: Task.ID, phaseId: Phase.ID)
          (implicit session: User.Session, scenario: Scenario): IO[Task.ID] =
    for {
      newTaskId <- copy(taskId, phaseId)
      _ <- delete(taskId)
    } yield newTaskId

  def move(taskId: Task.ID, container: ConcreteTarget)
          (implicit session: User.Session, scenario: Scenario): IO[Task.ID] = container match {
    case PhaseTarget(phaseId) => move(taskId, phaseId)
    case _ =>
      for {
        task <- get(taskId)
        newTaskId <- append(task, container)
        _ <- delete(taskId)
      } yield newTaskId
  }

  def move(taskId: Task.ID, subTaskId: Task.ID)
          (implicit session: User.Session, scenario: Scenario): IO[Task.ID] =
    move(taskId, taskId.target)

  def appendScript(phaseId: Phase.ID, title: String, taskType: String, script: String)
                  (implicit session: User.Session, scenario: Scenario): IO[Task.ID] =
    for {
      _ <- log.session.debug(s"xlr.tasks.appendScript(${phaseId.show}, $title, $taskType, $script)")
      resp <- http.session.post(server.api(_ ?/ "tasks" / "Applications" ++ phaseId.path / "tasks"),
        JsObject(
          "id" -> JsNull,
          "title" -> title.toJson,
          "type" -> taskType.toJson,
          "script" -> script.toJson
        ).toHttpEntity)
      content <- json.parse(resp)
      taskId <- JsUtils.readTaskId(sep = "/")(content).io
    } yield taskId

  def appendManual(phaseId: Phase.ID, title: String)
                  (implicit session: User.Session, scenario: Scenario): IO[Task.ID] =
    phases.appendTask(phaseId, title, "xlrelease.Task")

  def appendDeliveryTask(phaseId: Phase.ID, task: DeliveryTask)
                        (implicit session: User.Session, scenario: Scenario): IO[Task.ID] =
    append(JsObject(
      "id" -> JsNull,
      "title" -> task.title.toJson,
      "type" -> "xlrelease.CustomScriptTask".toJson,
      "pythonScript" -> task.toJson
    ),
      PhaseTarget(phaseId)
    )

  def appendGate(phaseId: Phase.ID, title: String, targets: Seq[Target])
                (implicit session: User.Session, scenario: Scenario): IO[(Task.ID, Seq[Dependency])] =
    for {
      taskId <- phases.appendTask(phaseId, title, "xlrelease.GateTask")
      dependencies <- targets.toList.map { target =>
        addDependency(taskId, target).map(id => Dependency(id, target))
      }.sequence
    } yield taskId -> dependencies

  def addDependency(gateTaskId: Task.ID, target: Target)
                   (implicit session: User.Session, scenario: Scenario): IO[Dependency.ID] =
    for {
      _ <- log.session.debug(s"xlr.tasks.addDependency(${gateTaskId.show}, ${target.show})")
      targetShortPath = target match {
        case vt: VariableTarget => ("${" + vt.variableKey + "}").asPath
        case ct: ConcreteTarget => ct.show.asPath
      }
      targetPath = Uri.Path.Empty ++ targetShortPath
      resp <- http.session.post(server.api(_ ?/ "tasks" / "Applications" ++ gateTaskId.path / "dependencies" ++ targetPath),
        JsObject(
          "taskId" -> ("Applications/" ++ gateTaskId.show).toJson,
          "targetId" -> target.show.toJson
        ).toHttpEntity
      )
      content <- json.parse(resp)
      dependencyId <- JsUtils.readDependencyId(sep = "/")(content).io
    } yield dependencyId


  def assignTo(taskId: Task.ID, assignee: User.ID)
              (implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.tasks.assignTo(${taskId.show}, $assignee")
      resp <- http.session.post(
        server.api(_ ?/ "tasks" / "Applications" ++ taskId.path / "assign" / assignee),
        JsNull.toHttpEntity
      )
      _ <- http.discard(resp)
    } yield ()

  def complete(taskId: Task.ID, comment: Option[String] = None)
              (implicit session: User.Session, scenario: Scenario): IO[Boolean] =
    for {
      _ <- log.session.debug(s"xlr.tasks.complete(${taskId.show}, $comment)")
      resp <- http.session.post(
        server.api(_ ?/ "tasks" / "Applications" ++ taskId.path / "complete"),
        comment.map(content => JsObject("comment" -> content.toJson)).getOrElse(JsObject.empty).toHttpEntity
      )
      content <- json.parse(resp)
    } yield JsUtils.matchesTaskStatus(TaskStatus.Completed)(content)

  def fail(taskId: Task.ID, comment: String)
          (implicit session: User.Session, scenario: Scenario): IO[Boolean] =
    for {
      _ <- log.session.debug(s"xlr.tasks.fail(${taskId.show}, $comment)")
      resp <- http.session.post(
        server.api(_ ?/ "tasks" / "Applications" ++ taskId.path / "fail"),
        JsObject("comment" -> comment.toJson).toHttpEntity
      )
      content <- json.parse(resp)
    } yield JsUtils.matchesTaskStatus(TaskStatus.Failed)(content)

  def retry(taskId: Task.ID, comment: String)
           (implicit session: User.Session, scenario: Scenario): IO[Boolean] =
    for {
      _ <- log.session.debug(s"xlr.tasks.retry(${taskId.show}, $comment)")
      resp <- http.session.post(
        server.api(_ ?/ "tasks" / "Applications" ++ taskId.path / "retry"),
        JsObject("comment" -> comment.toJson).toHttpEntity
      )
      content <- json.parse(resp)
    } yield JsUtils.matchesTaskStatus(TaskStatus.InProgress)(content)

  def skip(taskId: Task.ID, comment: String)
          (implicit session: User.Session, scenario: Scenario): IO[Comment.ID] = notImplemented("skip")

  def waitFor(taskId: Task.ID,
              expectedStatus: TaskStatus = TaskStatus.InProgress,
              interval: FiniteDuration = 5 seconds,
              retries: Option[Int] = Some(20))
             (implicit session: User.Session, scenario: Scenario): IO[Unit] = {
    for {
      _ <- log.session.debug(s"xlr.tasks.waitFor(${taskId.show}, $expectedStatus, $interval, $retries")
      _ <- control.until[Option[TaskStatus]](_.contains(expectedStatus), interval, retries) {
        getTaskStatus(taskId)
      }
    } yield ()
  }

  def getTaskStatus(taskId: Task.ID)
                   (implicit session: User.Session, scenario: Scenario): IO[Option[TaskStatus]] =
    for {
      _ <- log.session.debug(s"xlr.tasks.getTaskStatus(${taskId.show})")
      payload = JsObject("ids" -> Seq(taskId.show.replace("/", "-").stripPrefix("Applications-")).toJson)
      resp <- http.session.post(server.root(_ ?/ "tasks" / "poll"), payload.toHttpEntity)
      content <- json.parse(resp)
      taskStatus <- JsUtils.readFirstTaskStatus(content).io
    } yield taskStatus


  def getComments(taskId: Task.ID)
                 (implicit session: User.Session, scenario: Scenario): IO[Seq[Comment]] =
    for {
      _ <- log.session.debug(s"xlr.tasks.getComments(${taskId.show})")
      resp <- http.session.get(server.api(_ ?/ "tasks" / "Applications" ++ taskId.path))
      content <- json.parse(resp)
      comments <- JsUtils.readComments(content).io
    } yield comments

  def comment(taskId: Task.ID, comment: String)
             (implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.tasks.comment(${taskId.show}, $comment)")
      resp <- http.session.post(server.api(_ ?/ "tasks" / "Applications" ++ taskId.path / "comment"),
        JsObject(
          "comment" -> comment.toJson
        ).toHttpEntity
      )
      _ <- http.discard(resp)
    } yield ()

  // TODO: for now, we assume zip file only, and we return a string (Attachment.ID)
  def addAttachment(taskId: Task.ID, file: File)
                   (implicit session: User.Session, scenario: Scenario): IO[String] =
    for {
      _ <- log.session.debug(s"xlr.tasks.addAttachment(${taskId.show}, ${file.getAbsolutePath})")
      resp <- http.session.post(server.api(_ ?/ "tasks" / "Applications" ++ taskId.path / "attachments"),
        MultipartZip(file).toEntity
      )
      content <- json.parse(resp)
      attachmentId <- JsUtils.readFirstId(content).io
    } yield attachmentId

  private def notImplemented[A](name: String): IO[A] = control.fail(s"Not implemented: $name")
}
