package com.xebialabs.xlrelease.stress.api.xlr

import akka.http.scaladsl.model.{StatusCodes, Uri}
import cats.Show
import cats.effect.IO
import cats.implicits._
import com.github.nscala_time.time.Imports.DateTime
import com.xebialabs.xlrelease.stress.{Scenario, api}
import com.xebialabs.xlrelease.stress.config.XlrServer
import com.xebialabs.xlrelease.stress.domain._
import com.xebialabs.xlrelease.stress.protocol.{CreateReleaseArgs, DateFormat}
import com.xebialabs.xlrelease.stress.utils.HttpHelpers.JsonToHttpEntity
import com.xebialabs.xlrelease.stress.utils.IOHelpers.EitherToIO
import com.xebialabs.xlrelease.stress.utils.JsUtils
import spray.json._

import scala.concurrent.duration._
import scala.language.postfixOps


class Releases(server: XlrServer)
              (implicit
               http: api.http.Client with api.http.Session,
               control: api.control.Control with api.control.Flow,
               log: api.log.Logging with api.log.Session,
               json: api.json.JsonParser) extends DefaultJsonProtocol with DateFormat {
  def createFromTemplate(templateId: Template.ID, createReleaseArgs: CreateReleaseArgs)
                        (implicit session: User.Session, scenario: Scenario): IO[Release.ID] =
    for {
      _ <- log.session.debug(s"xlr.releases.createFromTemplate(${templateId.show}, ${createReleaseArgs.show})")
      resp <- http.session.post(server.api(_ ?/ "templates" / "Applications" ++ templateId.path / "create"), createReleaseArgs.toJson.toHttpEntity)
      content <- json.parse(resp)
      releaseId <- JsUtils.readIdString(content).io
    } yield Release.ID(releaseId)

  def create(title: String, scriptUser: Option[User] = None)
            (implicit session: User.Session, scenario: Scenario): IO[Phase.ID] = {
    val user: User = scriptUser.getOrElse(session.user)
    for {
      _ <- log.session.debug(s"xlr.releases.createRelease($title, ${scriptUser.map(_.show)})")
      resp <- http.session.post(server.root(_ ?/ "releases"),
        JsObject(
          "templateId" -> JsNull,
          "title" -> title.toJson,
          "scheduledStartDate" -> DateTime.now.toJson,
          "dueDate" -> (DateTime.now plusHours 5).toJson,
          "type" -> "xlrelease.Release".toJson,
          "owner" -> JsObject(
            "username" -> session.user.username.toJson
          ),
          "scriptUsername" -> JsObject(
            "username" -> user.username.toJson
          ),
          "scriptUserPassword" -> user.password.toJson
        ).toHttpEntity)
      content <- json.parse(resp)
      phaseId <- JsUtils.readFirstPhaseId(sep = "-")(content).io
    } yield phaseId
  }

  def get(releaseId: Release.ID)
         (implicit session: User.Session, scenario: Scenario): IO[JsObject] =
    for {
      _ <- log.session.debug(s"xlr.releases.get(${releaseId.show})")
      resp <- http.session.get(server.api(_ ?/ "releases" / "Applications" ++ releaseId.path))
      releaseJson <- json.parse(resp)
      releaseJsonObj <- JsUtils.jsObject(releaseJson).io
    } yield releaseJsonObj

  def getArchived(releaseId: Release.ID)
                 (implicit session: User.Session, scenario: Scenario): IO[JsObject] =
    for {
      _ <- log.session.debug(s"xlr.releases.getArchived(${releaseId.show})")
      resp <- http.session.get(server.api(_ ?/ "releases" / "archived" / "Applications" ++ releaseId.path))
      releaseJson <- json.parse(resp)
      releaseJsonObj <- JsUtils.jsObject(releaseJson).io
    } yield releaseJsonObj

  def start(releaseId: Release.ID)
           (implicit session: User.Session, scenario: Scenario): IO[Release.ID] =
    for {
      _ <- log.session.debug(s"xlr.releases.start(${releaseId.show})")
      resp <- http.session.post(server.api(_ ?/ "releases" / "Applications" ++ releaseId.path / "start"), JsNull.toHttpEntity)
      _ <- http.discard(resp)
    } yield releaseId

  def abort(releaseId: Release.ID)
           (implicit session: User.Session, scenario: Scenario): IO[Release.ID] =
    for {
      _ <- log.session.debug(s"xlr.releases.abort(${releaseId.show})")
      resp <- http.session.post(server.api(_ ?/ "releases" / "Applications" ++ releaseId.path / "abort"), JsNull.toHttpEntity)
      _ <- http.discard(resp)
    } yield releaseId

  def getTasksByTitle(releaseId: Release.ID, taskTitle: String, phaseTitle: Option[String] = None)
                     (implicit session: User.Session, scenario: Scenario): IO[Seq[Task.ID]] = {
    val baseQuery = Uri.Query(
      "releaseId" -> s"Applications/${releaseId.show}",
      "taskTitle" -> taskTitle
    )
    val query = phaseTitle.fold(baseQuery) { pt =>
      ("phaseTitle" -> pt) +: baseQuery
    }
    for {
      _ <- log.session.debug(s"xlr.releases.getTasksByTitle(${releaseId.id}, $taskTitle, $phaseTitle)")
      resp <- http.session.get(server.api(_ ?/ "tasks" / "byTitle").withQuery(query))
      content <- json.parse(resp)
      taskIds <- JsUtils.readTaskIds(sep = "/")(content).io
    } yield taskIds
  }

  def search(title: Option[String] = None,
             tags: Set[String] = Set.empty,
             statuses: Set[ReleaseStatus] = Set.empty,
             onlyArchived: Boolean = false)
            (page: Int = 0, resultsPerPage: Int = 100)
            (implicit session: User.Session, scenario: Scenario): IO[JsArray] = {
    val query = Uri.Query(
      "page" -> page.toString,
      "resultsPerPage" -> resultsPerPage.toString
    )
    val filters: Map[String, Option[JsValue]] = statuses.map { status =>
      status.name -> Some(true.toJson) // TODO: check if we need also the 'false' entries and what to do with 'active' and 'inactive' switches.
    }.toMap ++ Map(
      "title" -> title.map(_.toJson),
      "tags" -> Option(tags).filter(_.nonEmpty).map(_.toJson),
      "onlyArchived" -> Some(onlyArchived.toJson)
    )
    val payload = JsObject(filters.collect { case (k, Some(v)) => k -> v})
    for {
      _ <- log.session.debug(s"xlr.releases.search($title, $tags, $statuses, $onlyArchived)")
      resp <- http.session.post(server.api(_ ?/ "releases" / "search").withQuery(query), payload.toHttpEntity)
      content <- json.parse(resp)
      array <- JsUtils.jsArray(content).io
    } yield array
  }

  def waitFor(releaseId: Release.ID, status: ReleaseStatus, interval: FiniteDuration = 5 seconds, retries: Option[Int] = None)
             (implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.releases.waitFor(${releaseId.show}, $status, $interval, $retries)")
      _ <- control.until[ReleaseStatus](_ == status, interval, retries) {
        getReleaseStatus(releaseId).map(_._1)
      }
    } yield ()

  def getReleaseStatus(releaseId: Release.ID)
                      (implicit session: User.Session, scenario: Scenario): IO[(ReleaseStatus, Map[Task, TaskStatus])] =
    for {
      _ <- log.session.debug(s"xlr.releases.getReleaseStatus(${releaseId.show})")
      resp <- http.session.get(server.api(_ ?/ "releases" / "Applications" ++ releaseId.path))
      content <- json.parse(resp)
      releaseStatus <- JsUtils.readReleaseStatus(content).io
      tasks <- JsUtils.readTasksAndStatuses(content).io
    } yield (releaseStatus, tasks)

  def waitUntilArchived(releaseId: Release.ID, interval: FiniteDuration = 5 seconds, retries: Option[Int] = None)
                       (implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.releases.waitForArchived(${releaseId.show}, $interval, $retries)")
      _ <- control.until[Boolean](_ == true, interval, retries) {
        getReleaseIsArchived(releaseId)
      }
    } yield ()

  def getReleaseIsArchived(releaseId: Release.ID)
                          (implicit session: User.Session, scenario: Scenario): IO[Boolean] =
    for {
      _ <- log.session.debug(s"xlr.releases.getReleaseIsArchived(${releaseId.show})")
      resp <- http.session.get(server.api(_ ?/ "releases" / "archived" / "Applications" ++ releaseId.path))
      status = resp.status
      _ <- http.discard(resp)
    } yield status == StatusCodes.OK

  def createVariable[A](releaseId: Release.ID, variable: Variable[A])
                       (implicit session: User.Session, scenario: Scenario, showValue: Show[A]): IO[Variable.ID] =
    for {
      _ <- log.session.debug(s"xlr.releases.createVariable(${releaseId.show}, ${variable.show})")
      resp <- http.session.post(server.api(_ ?/ "releases" / "Applications" ++ releaseId.path / "variables"), variable.toJson.toHttpEntity)
      content <- json.parse(resp)
      variableId <- JsUtils.readIdString(content).io
    } yield Variable.ID(Some(releaseId), variableId.split("/").last)

}
