package com.xebialabs.xlrelease.stress.api.xlr

import akka.http.scaladsl.model.Uri
import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.{Scenario, api}
import com.xebialabs.xlrelease.stress.config.XlrServer
import com.xebialabs.xlrelease.stress.utils.HttpHelpers.JsonToHttpEntity
import com.xebialabs.xlrelease.stress.utils.IOHelpers.EitherToIO
import com.xebialabs.xlrelease.stress.domain._
import com.xebialabs.xlrelease.stress.utils.JsUtils
import spray.json._


class Phases(server: XlrServer)
            (implicit
             http: api.http.Client with api.http.Session,
             log: api.log.Logging with api.log.Session,
             json: api.json.JsonParser) extends DefaultJsonProtocol {

  def appendPhase(releaseId: Release.ID)
                      (implicit session: User.Session, scenario: Scenario): IO[Phase.ID] = {
    for {
      _ <- log.session.debug(s"xlr.phases.appendPhase(${releaseId.show})")
      resp <- http.session.post(server.api(_ ?/ "phases" / "Applications" ++ releaseId.path / "phase"),
        JsObject(
          "id" -> JsNull,
          "type" -> "xlrelease.Phase".toJson,
          "title" -> "Phase".toJson,
          "flagStatus" -> "OK".toJson,
          "overdueNotified" -> false.toJson,
          "tasks" -> List.empty[String].toJson,
          "release" -> "Applications/${releaseId.path}".toJson,
          "status" -> "PLANNED".toJson,
          "color" -> "#00FF00".toJson
        ).toHttpEntity
      )
      content <- json.parse(resp)
      phaseId <- (JsUtils.readIdString(content) >>= JsUtils.parsePhaseId()).io
    } yield phaseId
  }


  def appendTask(phaseId: Phase.ID, title: String, taskType: String)
                (implicit session: User.Session, scenario: Scenario): IO[Task.ID] =
    for {
      _ <- log.session.debug(s"xlr.phases.appendToPhase(${phaseId.show}, $title, $taskType)")
      resp <- http.session.post(server.api(_ ?/ "tasks" / "Applications" ++ phaseId.path / "tasks"),
        JsObject(
          "id" -> JsNull,
          "title" -> title.toJson,
          "type" -> taskType.toJson
        ).toHttpEntity)
      content <- json.parse(resp)
      taskId <- JsUtils.readTaskId(sep = "/")(content).io
    } yield taskId

  def insertTask(phaseId: Phase.ID, title: String, taskType: String, position: Int = 0)
                (implicit session: User.Session, scenario: Scenario): IO[Task.ID] = {
    require(position >= 0)
    for {
      _ <- log.session.debug(s"xlr.phases.insertInPhase(${phaseId.show}, $title, $taskType, $position)")
      resp <- http.session.post(
        server.api(_ ?/ "phases" / "Applications" ++ phaseId.path / "tasks").withQuery(
          Uri.Query("position" -> position.toString)
        ),
        JsObject(
          "id" -> JsNull,
          "title" -> title.toJson,
          "type" -> taskType.toJson
        ).toHttpEntity
      )
      content <- json.parse(resp)
      taskId <- JsUtils.readTaskId(sep = "/")(content).io
    } yield taskId
  }
}