package com.xebialabs.xlrelease.stress.api.xlr

import akka.http.scaladsl.model.Uri
import cats.effect.IO
import com.xebialabs.xlrelease.stress.{Scenario, api}
import com.xebialabs.xlrelease.stress.config.XlrServer
import com.xebialabs.xlrelease.stress.domain.{Pattern, Stage, TrackedItem, Transition, User}
import com.xebialabs.xlrelease.stress.protocol.DateFormat
import com.xebialabs.xlrelease.stress.utils.JsUtils
import com.xebialabs.xlrelease.stress.utils.IOHelpers.EitherToIO
import com.xebialabs.xlrelease.stress.utils.HttpHelpers.JsonToHttpEntity
import spray.json._

class Patterns (server: XlrServer)
               (implicit
                http: api.http.Client with api.http.Session,
                control: api.control.Control with api.control.Flow,
                json: api.json.JsonParser,
                log: api.log.Logging with api.log.Session) extends DefaultJsonProtocol with DateFormat {

  def create(title: String,
             folderId: String)
            (implicit session: User.Session, scenario: Scenario): IO[Stage.ID] = {
    val payload = JsObject(
      "id" -> JsNull,
      "type" -> "delivery.Delivery".toJson,
      "status" -> "TEMPLATE".toJson,
      "title" -> title.toJson,
      "description" -> "test description".toJson,
      "folderId" -> ("Applications/" + folderId).toJson,
      "plannedDuration" -> JsNumber.zero,
      "releaseIds" -> JsArray.empty,
      "stages" -> JsArray.empty,
      "subscribers" -> JsArray.empty,
      "trackedItems" -> JsArray.empty
    )
    for {
      _ <- log.session.debug(s"xlr.patterns.create($title)")
      resp <- http.session.post(server.api(_ ?/ "delivery-patterns"), payload.toHttpEntity)
      content <- json.parse(resp)
      firstStageId <- JsUtils.readFirstStageId()(content).io
      _ <- log.session.debug(s"pattern content ${content.prettyPrint}")
    } yield firstStageId
  }

  def addTrackedItem(title: String, patternId: Pattern.ID)
                    (implicit session: User.Session, scenario: Scenario): IO[TrackedItem.ID] = {
    val payload = JsObject(
      "id" -> JsNull,
      "title" -> title.toJson,
      "type" -> "delivery.TrackedItem".toJson
    )
    for {
      _ <- log.session.debug(s"xlr.patterns.addTrackedItem($title)")
      resp <- http.session.post(server.api(_ ?/ "delivery-patterns" ++ patternId.path + "/tracked-items"), payload.toHttpEntity)
      content <- json.parse(resp)
      trackedItemId <- JsUtils.readIdString(content).io
    } yield TrackedItem.ID(trackedItemId)
  }

  def addStage(title: String,
               patternId: Pattern.ID,
               position: Number)
              (implicit session: User.Session, scenario: Scenario): IO[Stage.ID] = {
    val query = Uri.Query(
      "position" -> position.toString
    )
    val payload = JsObject(
      "id" -> JsNull,
      "title" -> title.toJson,
      "type" -> "delivery.Stage".toJson
    )
    for {
      _ <- log.session.debug(s"xlr.patterns.addStage($title)")
      resp <- http.session.post(server.api(_ ?/ "delivery-patterns" ++ patternId.path + "/stages").withQuery(query), payload.toHttpEntity)
      content <- json.parse(resp)
      stageId <- JsUtils.readIdString(content).io
    } yield Stage.ID(stageId)
  }

  def deleteStage(stageId: Stage.ID)
              (implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.patterns.deleteStage($stageId)")
      resp <- http.session.delete(server.api(_ ?/ "delivery-patterns" ++ stageId.path))
      _ <- http.discard(resp)
    } yield ()

  def addTransition(patternId: Pattern.ID,
                    fromStageId: Stage.ID,
                    title: String)
                   (implicit session: User.Session, scenario: Scenario): IO[Transition.ID] = {
    val payload = JsObject(
      "fromStageId" -> ("Applications/" + fromStageId).toJson,
      "id" -> JsNull,
      "title" -> title.toJson,
      "type" -> "delivery.ManualTransition".toJson
    )
    for {
      _ <- log.session.debug(s"xlr.patterns.addStage($title)")
      resp <- http.session.post(server.api(_ ?/ "delivery-patterns" ++ patternId.path / "transitions"), payload.toHttpEntity)
      content <- json.parse(resp)
      transitionId <- JsUtils.readIdString(content).io
    } yield Transition.ID(transitionId)

  }
}
