package com.xebialabs.xlrelease.stress.api.xlr

import akka.http.scaladsl.model.Uri
import cats.effect.IO
import com.xebialabs.xlrelease.stress.{Scenario, api}
import com.xebialabs.xlrelease.stress.config.XlrServer
import com.xebialabs.xlrelease.stress.domain.{Folder, Team, User}
import com.xebialabs.xlrelease.stress.utils.JsUtils
import com.xebialabs.xlrelease.stress.utils.HttpHelpers.JsonToHttpEntity
import com.xebialabs.xlrelease.stress.utils.IOHelpers.EitherToIO
import spray.json._


class Folders(server: XlrServer)
             (implicit
              http: api.http.Client with api.http.Session,
              log: api.log.Logging with api.log.Session,
              json: api.json.JsonParser) extends DefaultJsonProtocol {

  def createFolder(folder: Folder)
                  (implicit session: User.Session, scenario: Scenario): IO[Folder] =
    for {
      _ <- log.session.debug(s"xlr.folders.createFolder($folder)")
      parentFolderPath = folder.parent.map(_.id.path).getOrElse(Uri.Path.Empty)
      resp <- http.session.post(server.api(_ ?/ "folders" / "Applications" ++ parentFolderPath),
        JsObject(
          "title" -> folder.name.toJson,
          "type" -> "xlrelease.Folder".toJson,
          "id" -> JsNull
        ).toHttpEntity
      )
      content <- json.parse(resp)
      folderId <- JsUtils.readFolderId(sep = "/")(content).io
    } yield folder.copy(id = folderId)

  def deleteFolder(folderId: Folder.ID)
                  (implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.folders.deleteFolder($folderId)")
      resp <- http.session.delete(server.api(_ ?/ "folders" / "Applications" ++ folderId.path))
    } yield http.discard(resp)

  def renameFolder(folderId: Folder.ID, newName: String)
                  (implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.folders.renameFolder($folderId, $newName)")
      query = Uri.Query(
        "newName" -> newName
      )
      resp <- http.session.post(server.api(_ ?/ "folders" / "Applications" ++ folderId.path / "rename").withQuery(query), JsNull.toHttpEntity)
    } yield http.discard(resp)

  def moveFolder(folderId: Folder.ID, newParentId: Folder.ID)
                (implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.folders.moveFolder($folderId, $newParentId)")
      query = Uri.Query(
        "newParentId" -> newParentId.id
      )
      resp <- http.session.post(server.api(_ ?/ "folders" / "Applications" ++ folderId.path / "move").withQuery(query), JsNull.toHttpEntity)
    } yield http.discard(resp)

  def getTeams(folderId: Folder.ID)
              (implicit session: User.Session, scenario: Scenario): IO[Seq[Team]] =
    for {
      _ <- log.session.debug(s"xlr.folders.getTeams($folderId)")
      resp <- http.session.get(server.api(_ ?/ "folders" / "Applications" ++ folderId.path / "teams"))
      content <- json.parse(resp)
      teams <- JsUtils.readTeams(content).io
    } yield teams

  def setTeams(folderId: Folder.ID, teams: Seq[Team])
              (implicit session: User.Session, scenario: Scenario): IO[Map[String, String]] =
    for {
      _ <- log.session.debug(s"xlr.folders.setTeams($folderId, ${teams.map(_.teamName).mkString("[", ", ", "]")})")
      resp <- http.session.post(server.api(_ ?/ "folders" / "Applications" ++ folderId.path / "teams"), teams.map(_.toJson).toJson.toHttpEntity)
      content <- json.parse(resp)
      teamIds <- JsUtils.readTeamIds(content).io
    } yield teamIds

}
