package com.xebialabs.xlrelease.stress.api.xlr

import cats.effect.IO
import com.xebialabs.xlrelease.stress.config.XlrServer
import com.xebialabs.xlrelease.stress.domain.{Configuration, User, Variable}
import com.xebialabs.xlrelease.stress.utils.HttpHelpers.{IdAsUriPath, JsonToHttpEntity}
import com.xebialabs.xlrelease.stress.utils.IOHelpers.EitherToIO
import com.xebialabs.xlrelease.stress.utils.JsUtils
import com.xebialabs.xlrelease.stress.{Scenario, api}
import spray.json._


class Configurations(server: XlrServer)
                    (implicit
                     control: api.control.Control,
                     http: api.http.Client with api.http.Session,
                     log: api.log.Logging with api.log.Session,
                     json: api.json.JsonParser)
  extends DefaultJsonProtocol {
  self =>

  def setArchivingSettings(releaseAgeToDeleteFromJcr: Int, preArchivingEnabled: Boolean = true)
                          (implicit session: User.Session, scenario: Scenario): IO[Unit] = {
    for {
      _ <- log.session.debug(s"xlr.configurations.setArchivingSettings($releaseAgeToDeleteFromJcr, $preArchivingEnabled)")
      resp <- http.session.put(server.root(_ ?/ "settings" / "ci" / "Configuration" / "settings" / "ArchivingSettings"),
        JsObject(
          "id" -> "Configuration/settings/ArchivingSettings".toJson,
          "type" -> "xlrelease.ArchivingSettings".toJson,
          "releaseAgeToDeleteFromJcr" -> releaseAgeToDeleteFromJcr.toJson,
          "preArchivingEnabled" -> preArchivingEnabled.toJson
        ).toHttpEntity
      )
      content <- json.parse(resp)
      updatedAge <- JsUtils.getField("releaseAgeToDeleteFromJcr")(content).flatMap {
        case JsNumber(value) if value.isValidInt => Right(value.toIntExact)
        case other =>
          JsUtils.err[Int]("Expected integer", other, List("releaseAgeToDeleteFromJcr") )
      }.io
      enabled <- JsUtils.getBooleanField("preArchivingEnabled")(content).map(_.value).io
      _ <- if (enabled != preArchivingEnabled || updatedAge != releaseAgeToDeleteFromJcr) {
        control.error(new IllegalStateException("Failed to update ArchivingSettings"))
      } else {
        control.nop
      }
    } yield ()
  }

  def createGlobalVariable[T](variable: Variable[T])
                             (implicit session: User.Session, scenario: Scenario): IO[Variable.ID] =
    for {
      _ <- log.session.debug(s"xlr.configurations.createGlobalVariable($variable)")
      resp <- http.session.post(server.api(_ ?/ "config" / "Configuration" / "variables" / "global"),
        variable.toJson.toHttpEntity
      )
      content <- json.parse(resp)
      variableId <- JsUtils.readIdString(content).io
    } yield Variable.ID(None, variableId)

  def deleteGlobalVariable(variableId: Variable.ID)
                          (implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.configurations.deleteGlobalVariable($variableId)")
      resp <- http.session.delete(server.api(_ ?/ "config" ++ variableId.path))
      _ <- http.discard(resp)
    } yield ()

  def create(configuration: Configuration)
            (implicit session: User.Session, scenario: Scenario): IO[Configuration.ID] =
    for {
      _ <- log.session.debug(s"xlr.configurations.create($configuration)")
      resp <- http.session.post(server.api(_ ?/ "config"),
        configuration.body.toHttpEntity
      )
      content <- json.parse(resp)
      configurationId <- JsUtils.readIdString(content).io
    } yield configurationId

  def delete(configurationId: Configuration.ID)
            (implicit session: User.Session, scenario: Scenario): IO[Unit] =
    for {
      _ <- log.session.debug(s"xlr.configurations.delete($configurationId)")
      resp <- http.session.delete(server.api(_ ?/ "config" ++ configurationId.asPath))
      _ <- http.discard(resp)
    } yield ()

}
