package com.xebialabs.xlrelease.stress.api.metric

import java.io.File
import java.nio.file.Paths

import akka.http.scaladsl.model.Uri
import akka.http.scaladsl.model.headers.{Authorization, OAuth2BearerToken}
import akka.stream.Materializer
import akka.stream.scaladsl.FileIO
import cats.effect.{ContextShift, IO}
import cats.implicits._
import com.xebialabs.xlrelease.stress.config.GrafanaConfig
import com.xebialabs.xlrelease.stress.config.GrafanaConfig.{DashboardConfig, DashboardsConfig, PanelConfig}
import com.xebialabs.xlrelease.stress.utils.IOHelpers.FutureToIO
import com.xebialabs.xlrelease.stress.{Scenario, api}
import org.joda.time.{DateTime, Duration}

import scala.concurrent.ExecutionContext

class GrafanaClient(serverConfig: GrafanaConfig.ServerConfig)
                   (implicit
                    control: api.control.Control,
                    http: api.http.Client with api.http.Session,
                    log: api.log.Logging,
                    json: api.json.JsonParser,
                    ec: ExecutionContext,
                    m: Materializer,
                    cs: ContextShift[IO]) {

  // TODO: https://grafana.com/docs/http_api/auth/ => authentication using API Token
  def fetchGraph(dashboard: DashboardConfig, panel: PanelConfig, from: DateTime, to: DateTime)
                (implicit
                 scenario: Scenario,
                 dashboardsConfig: DashboardsConfig): IO[File] = {
    val uri = serverConfig.server.withPath(path(dashboard)).withQuery(query(dashboard, panel, from -> to))
    for {
      _ <- log.debug(s"metric.grafana.fetchGraph(${dashboard.name}, ${panel.name}, $from, $to)")
      _ <- log.info(uri.toString)
      resp <- http.get(
        uri,
        headers = Authorization(OAuth2BearerToken(token = serverConfig.token)) :: Nil
      )
      filename = s"${dashboard.name}_${panel.name}.png"
      path = serverConfig.downloadDir.resolve(scenario.name).resolve(filename)
      file <- IO {
        val f = path.toFile
        f.getParentFile.mkdirs()
        f.createNewFile()
        f
      }
      _ <- resp.entity.dataBytes.runWith(FileIO.toPath(Paths.get(file.getAbsolutePath))).io
    } yield file
  }

  protected def path(dashboard: DashboardConfig)
                    (implicit dashboardsConfig: DashboardsConfig): Uri.Path =
    serverConfig.rootPath / "render" / "d-solo" / dashboard.id / dashboard.name

  protected def query(dashboard: DashboardConfig, panel: PanelConfig, range: (DateTime, DateTime))
                     (implicit dashboardsConfig: DashboardsConfig): Uri.Query = {
    val (from, to) = {
      val duration = new Duration(range._1, range._2)
      if (duration.getStandardMinutes < 5) {
        val diff = Duration.standardMinutes(5) minus duration
        range._1.minus(diff) -> range._2
      } else {
        range
      }
    }
    val base = Uri.Query(
      "width" -> dashboardsConfig.width.toString,
      "height" -> dashboardsConfig.height.toString,
      "orgId" -> dashboardsConfig.orgId.toString,
      "tz" -> dashboardsConfig.timezone,
      "from" -> from.getMillis.toString,
      "to" -> to.getMillis.toString,
      "panelId" -> panel.id.toString
    )
     dashboard.parameters.foldLeft(base) { case (query, (key, value)) =>
       (s"var-$key" -> value) +: query
     }
  }
}
