package com.xebialabs.xlrelease.stress

import cats.Show
import cats.effect.IO
import cats.implicits._
import com.xebialabs.xlrelease.stress.config.ReportingXlrConfig
import com.xebialabs.xlrelease.stress.runner.Runner
import play.api.libs.json._

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}

trait Scenario {
  type Params

  var parent: Option[Scenario] = None

  def name: String

  def fullName: String = s"${parent.map(_.fullName + ": ").getOrElse("")}$name"

  val api: API

  def setup: IO[Params]

  def program(params: Params): IO[Unit]

  def cleanup(params: Params): IO[Unit]

  implicit val showParams: Show[Params]

  implicit lazy val self: Scenario = this

}

object Scenario {

  abstract class ScenarioMaker[S <: Scenario, B](val name: String) {
    type Params = B

    val defaultParameters: JsObject = Json.obj()

    implicit val readParameters: Reads[B]

    def scenario(parameters: B)
                (implicit api: API): S

    def runJsonFuture(parameters: JsValue, setupTimeout: FiniteDuration, programTimeout: FiniteDuration)
                     (implicit ec: ExecutionContext, reportingConfig: ReportingXlrConfig, api: API): JsResult[Future[Unit]] = {
      parameters.validate[B](readParameters).map { p =>
        implicit val report: ReportingAPI = new ReportingAPI(api, reportingConfig)
        implicit val s: S = scenario(p)
        val preamble = for {
          _ <- api.log.info(s"Scenario instance created")
          _ <- api.log.info(s"parameters: ${Json.prettyPrint(parameters)}")
        } yield ()
        preamble.unsafeToFuture() >> Runner.runFuture(s, setupTimeout, programTimeout)
      }
    }
  }

}
