package com.xebialabs.xlrelease.stress.scenarios

import cats.Show
import cats.implicits._
import com.xebialabs.xlrelease.stress.config.XlrConfig
import com.xebialabs.xlrelease.stress.domain.Member.UserMember
import com.xebialabs.xlrelease.stress.domain.Permission.ViewFolder
import com.xebialabs.xlrelease.stress.domain._
import com.xebialabs.xlrelease.stress.dsl.DSL
import com.xebialabs.xlrelease.stress.dsl.libs.xlr.protocol.CreateReleaseArgs
import com.xebialabs.xlrelease.stress.scenarios.RunReleasesSequentially._
import freestyle.free._
import freestyle.free.implicits._

import scala.concurrent.duration._
import scala.io.Source
import scala.language.postfixOps

object RunReleasesSequentially {
  type Params = (User, Template.ID, Folder.ID, Variable.ID)

  // Release stops if a task needs manual intervention or if release finishes (is completed or aborted)
  type ReleaseStopReason = Either[(Task, TaskStatus), ReleaseStatus]
}

// This scenario is used from xlr-risk-predictions-plugin, so please check there before removing/changing.
case class RunReleasesSequentially(templateGroovyFile: String,
                                   applicationName: String,
                                   numReleases: Int,
                                   globalVariable: (String, Map[String, String]),
                                   abortEveryNthRelease: Int,
                                   durationManualTask: FiniteDuration = new FiniteDuration(60, SECONDS),
                                   durationBeforeRetry: FiniteDuration = new FiniteDuration(40, SECONDS),
                                   runCleanup: Boolean = true
                                  )
                                  (implicit val config: XlrConfig,
                                   val _api: DSL[DSL.Op])
  extends Scenario[Params] with ScenarioUtils {

  val name: String = "Run a release with some automatic and manual tasks; retry failing tasks; " +
    "or abort the whole release; do that many times"

  private val groovyTemplate: String = Source.fromResource(templateGroovyFile)
    .getLines()
    .mkString("\n")

  override def setup: Program[Params] =
    api.xlr.users.admin() flatMap { implicit session =>
      for {
        _ <- api.log.info(s"creating folder $applicationName")
        folderId <- api.xlr.folders.createFolder(Folder(applicationName, None, null))
        folder = Folder(applicationName, None, folderId)
        _ <- api.log.info("setting up a user and team")
        user <- createUser(applicationName)
        _ <- setupTeams(user, folder)
        releaseId <- createReleaseFromGroovy(
          s"Create template in $applicationName from groovy file $templateGroovyFile",
          groovyTemplate,
          Map(
            "###ADMIN_PASSWORD###" -> session.user.password,
            "###FOLDER_NAME###" -> folder.name
          )
        )
        templateId = Template.ID(releaseId.id)
        _ <- api.log.info(s"creating global variable ${globalVariable._1}")
        variableId <- api.xlr.configurations.createGlobalVariable(
          Variable.mapStringString(
            id = Variable.ID.empty,
            key = globalVariable._1,
            value = Option(globalVariable._2)
          )
        )
        _ <- api.log.info("setup complete!")
      } yield (user, templateId, folderId, variableId)
    }

  override def program(params: Params): Program[Unit] = params match {
    case (user, templateId, _, _) =>
      (1 to numReleases).toList.map { releaseNumber =>
        runRelease(user, templateId, releaseNumber)
      }.sequence[Program, Unit].map(_ => ())
  }

  override def cleanup(params: Params): Program[Unit] = if (runCleanup) {
    params match {
      case (user, _, folderId, variableId) =>
        api.xlr.users.admin() flatMap { implicit session =>
          for {
            _ <- api.log.info("Cleaning up folder, user and global variable")
            _ <- api.xlr.configurations.deleteGlobalVariable(variableId)
            _ <- api.xlr.users.deleteUser(user.username)
            _ <- api.xlr.folders.deleteFolder(folderId)
          } yield ()
        }
    }
  } else {
    api.ok(())
  }

  implicit val showParams: Show[Params] = {
    case (user, templateId, folderId, variableId) => s"$user, $templateId, $folderId, $variableId"
  }

  private def runRelease(user: User, templateId: Template.ID, releaseNumber: Int): Program[Unit] =
    api.xlr.users.login(user) flatMap { implicit session =>
      for {
        _ <- api.log.info(s"logged in as ${user.username}...")
        version = s"$releaseNumber.0.0"
        releaseTitle = s"$applicationName $version"
        _ <- api.log.info(s"Creating release $releaseTitle from template")
        releaseId <- api.xlr.releases.createFromTemplate(templateId, CreateReleaseArgs(
          title = releaseTitle,
          variables = Map("version" -> version)
        ))
        _ <- api.log.info(s"Starting release $releaseId")
        _ <- api.xlr.releases.start(releaseId)

        _ <- api.log.info("Waiting for release to stop")
        stopReason <- waitForReleaseToStopOrComplete(releaseId)

        _ <- if (stopReason.isRight) {
          api.log.info(s"Release has finished with status ${stopReason.right.get}: $releaseId"): Program[Unit]
        } else {
          if (releaseNumber % abortEveryNthRelease == 0) {
            (api.log.info(s"Aborting release #$releaseNumber as it is multiple of $abortEveryNthRelease"): Program[Unit]) >>
              api.xlr.releases.abort(releaseId)
          } else {
            retryAndCompleteUntilReleaseFinishes(releaseId, stopReason)
          }
        }
      } yield ()
    }

  private def retryAndCompleteUntilReleaseFinishes(releaseId: Release.ID, stopReason: ReleaseStopReason)
                                                  (implicit session: User.Session): Program[Unit] =
    stopReason match {
      case Right(status) =>
        api.log.info(s"Release has finished with status $status: $releaseId")

      case Left((task, status)) =>

        val completeOrRetry: Program[Unit] = status match {
          case TaskStatus.Failed =>
            for {
              _ <- api.log.info(s"Task ${task.title} has failed, assigning it and retrying after a while")
              _ <- api.xlr.tasks.assignTo(task.id, session.user.username)
              _ <- api.control.sleep(durationBeforeRetry)
              _ <- api.xlr.tasks.retry(task.id, "Something went wrong, I'll just retry")
            } yield ()
          case _ =>
            for {
              _ <- api.log.info(s"Completing task ${task.title} after a while")
              _ <- api.control.sleep(durationManualTask)
              _ <- api.xlr.tasks.complete(task.id, Some("Done!"))
            } yield ()
        }

        for {
          _ <- completeOrRetry
          _ <- api.log.info("Waiting for release to stop again")
          nextStopReason <- waitForReleaseToStopOrComplete(releaseId)
          _ <- retryAndCompleteUntilReleaseFinishes(releaseId, nextStopReason)
        } yield ()
    }

  private def setupTeams(user: User, folder: Folder)(implicit session: User.Session): Program[Unit] =
    for {
      _ <- api.log.info("getting folder teams...")
      teams <- api.xlr.folders.getTeams(folder.id)
      teamsMap = teams.map(t => t.teamName -> t).toMap
      folderOwner = teamsMap("Folder Owner") match {
        case team => team.copy(members = team.members :+ UserMember(user.username))
      }
      templateOwner = teamsMap("Template Owner") match {
        case team => team.copy(members = team.members :+ UserMember(user.username))
      }
      releaseAdmin = teamsMap("Release Admin") match {
        case team => team.copy(members = team.members :+ UserMember(user.username))
      }
      additionalTeam = Team(
        folder.name,
        members = Seq(UserMember(user.username)),
        permissions = (Team.templateOwnerPermissions ++ Team.releaseAdminPermissions + ViewFolder).toSeq
      )
      _ <- api.log.info("setting up teams:")
      _ <- api.log.info(folderOwner.show)
      _ <- api.log.info(templateOwner.show)
      _ <- api.log.info(releaseAdmin.show)
      _ <- api.log.info(additionalTeam.show)
      _ <- api.xlr.folders.setTeams(folder.id, Seq(folderOwner, templateOwner, releaseAdmin, additionalTeam))
      _ <- api.log.info("folder teams setup correctly")
    } yield ()

  private def createUser(folderName: String): Program[User] = {
    val username = s"${folderName.toLowerCase}_dev"
    val user = User(username, s"$folderName Dev", "", username)
    for {
      _ <- api.log.info(s"Creating user $username...")
      userId <- api.xlr.users.createUser(user)
    } yield user.copy(username = userId)
  }

  private def waitForReleaseToStopOrComplete(releaseId: Release.ID, interval: FiniteDuration = 5 seconds,
                                             retries: Option[Int] = Some(20))
                                            (implicit session: User.Session): Program[ReleaseStopReason] =
    for {
      _ <- api.log.debug(s"xlr.releases.waitForReleaseToStopOrComplete($releaseId, $interval, $retries)")
      stopReason <- api.lib.control.until[Option[ReleaseStopReason]](_.isDefined, interval, retries) {
        checkIfStopped(releaseId)
      }
    } yield stopReason.get

  private def checkIfStopped(releaseId: Release.ID)
                            (implicit session: User.Session): Program[Option[ReleaseStopReason]] =
    api.xlr.releases.getReleaseStatus(releaseId).map {
      case (status, _) if status == ReleaseStatus.Completed || status == ReleaseStatus.Aborted =>
        Some(status.asRight)
      case (_, statuses) if getFirstTaskWaitingForUser(statuses).isDefined =>
        Some(getFirstTaskWaitingForUser(statuses).get.asLeft)
      case _ =>
        None
    }

  private def getFirstTaskWaitingForUser(tasks: Map[Task, TaskStatus]): Option[(Task, TaskStatus)] =
    tasks.find {
      case (task, TaskStatus.InProgress) if task.isManual => true
      case (_, TaskStatus.WaitingForInput) => true
      case (_, TaskStatus.Failed) => true
      case _ => false
    }

}
