package com.xebialabs.xlrelease.stress.dsl.libs.xlr

import akka.http.scaladsl.model.Uri
import com.xebialabs.xlrelease.stress.config.XlrServer
import com.xebialabs.xlrelease.stress.domain.{Folder, Team, User}
import com.xebialabs.xlrelease.stress.dsl.DSL
import com.xebialabs.xlrelease.stress.utils.JsUtils
import freestyle.free._
import spray.json._


class Folders[F[_]](server: XlrServer)(implicit protected val _api: DSL[F])
  extends XlrLib[F] with DefaultJsonProtocol { self =>

  def createFolder(folder: Folder)
                  (implicit session: User.Session): Program[Folder.ID] =
    for {
      _ <- log.debug(s"xlr.folders.createFolder($folder)")
      parentFolderPath = folder.parent.map(_.id.path).getOrElse(Uri.Path.Empty)
      resp <- lib.http.json.post(server.api(_ ?/ "folders" / "Applications" ++ parentFolderPath),
        JsObject(
          "title" -> folder.name.toJson,
          "type" -> "xlrelease.Folder".toJson,
          "id" -> JsNull
        )
      )
      content <- api.http.parseJson(resp)
      folderId <- lib.json.read(JsUtils.readFolderId(sep = "/"))(content)
    } yield folderId

  def deleteFolder(folderId: Folder.ID)
                  (implicit session: User.Session): Program[Unit] =
    for {
      _ <- log.debug(s"xlr.folders.deleteFolder($folderId)")
      resp <- lib.http.delete(server.api(_ ?/ "folders" / "Applications" ++ folderId.path))
      _ <- api.http.discard(resp)
    } yield ()

  def getTeams(folderId: Folder.ID)
              (implicit session: User.Session): Program[Seq[Team]] =
    for {
      _ <- log.debug(s"xlr.folders.getTeams($folderId)")
      content <- lib.http.json.get(server.api(_ ?/ "folders" / "Applications" ++ folderId.path / "teams"))
      teams <- lib.json.read(JsUtils.readTeams)(content)
    } yield teams

  def setTeams(folderId: Folder.ID, teams: Seq[Team])
              (implicit session: User.Session): Program[Map[String, String]] =
    for {
      _ <- log.debug(s"xlr.folders.setTeams($folderId, ${teams.map(_.teamName).mkString("[", ", ", "]")})")
      resp <- lib.http.json.post(server.api(_ ?/ "folders" / "Applications" ++ folderId.path / "teams"), teams.map(_.toJson).toJson)
      content <- api.http.parseJson(resp)
      teamIds <- lib.json.read(JsUtils.readTeamIds)(content)
    } yield teamIds

}
