package com.xebialabs.xlrelease.stress.domain

import akka.http.scaladsl.model.Uri
import cats.Show
import cats.implicits._

sealed trait Target
object Target {
  case class VariableTarget(variableId: Variable.ID) extends Target

  sealed trait ConcreteTarget extends Target
  case class ReleaseTarget(releaseId: Release.ID) extends ConcreteTarget
  case class PhaseTarget(phaseId: Phase.ID) extends ConcreteTarget
  case class TaskTarget(taskId: Task.ID) extends ConcreteTarget

  implicit class ConcreteTargetOps(val target: ConcreteTarget) extends AnyVal {
    def asStringId: String = target match {
      case ReleaseTarget(releaseId) => releaseId.show
      case PhaseTarget(phaseId) => phaseId.show
      case TaskTarget(taskId) => taskId.show
    }

    def path: Uri.Path = asPath(asStringId)
  }

  implicit class ReleaseTargetOps(val releaseId: Release.ID) extends AnyVal {
    def target: ReleaseTarget = ReleaseTarget(releaseId)
  }

  implicit class PhaseTargetOps(val phaseId: Phase.ID) extends AnyVal {
    def target: PhaseTarget = PhaseTarget(phaseId)
  }

  implicit class TaskTargetOps(val taskId: Task.ID) extends AnyVal {
    def target: TaskTarget = TaskTarget(taskId)
  }

  implicit val showConcreteTarget: Show[ConcreteTarget] = ct => ct.asStringId

  implicit val showVariableTarget: Show[VariableTarget] = vt => vt.variableId.show

  implicit val showTarget: Show[Target] = {
    case vt: VariableTarget => showVariableTarget.show(vt)
    case ct: ConcreteTarget => showConcreteTarget.show(ct)
  }
}