import akka.http.javadsl.model.HttpRequest
import akka.http.javadsl.model.headers.HttpCredentials
import com.xebialabs.impact.api.BaseCrawlerMessage
import com.xebialabs.impact.api.CrawlerCredentialsDTO
import com.xebialabs.impact.api.PushCrawlerUMMessagesRequestVM
import com.xebialabs.impact.api.SetContextKeyRequest
import com.xebialabs.impact.wave.scripting.PluginInterface
import com.xebialabs.impact.wave.scripting.http.WaveContext

class NewHaveSeenContextValue {
    String haveSeen
    String contextKeyToUse
}

class CrawlEntity {
    String url
}

class CrawlEntityMessage extends BaseCrawlerMessage {
    String url
}

class CrawlEntitiesRequest {
    String previousLastUpdateDate
    String contextKeyToUse
    String url;
}

PluginInterface.map(CrawlEntities).into(CrawlEntitiesRequest).via({ crawlEntities, ctx ->
    def request = new CrawlEntitiesRequest()
    request.contextKeyToUse = crawlEntities.contextKeyToUse
    since = crawlEntities.context == null ? null : crawlEntities.context.get(crawlEntities.contextKeyToUse)
    request.url = "webservice/v2.0/artifact" +
            "?order=LastUpdateDate%20asc" +
            "&workspace=/workspace/" + crawlEntities.workspaceId +
            "&fetch=true" +
            "&start=1" +
            "&pagesize=" + calculatePageSize(ctx) + (
            (since == null || since.isEmpty()) ?
                    "" :
                    ("&query=(LastUpdateDate%20>=%20" + since + ")"))
    request.previousLastUpdateDate = since
    ctx.emit(request)
})


PluginInterface.request(CrawlEntitiesRequest).into(CrawlEntity, NewHaveSeenContextValue, DecideToNack)
        .mapToRequest({ crawlEntities, ctx ->
    CrawlerCredentialsDTO credentials = ctx.crawlerMessageWithCredentials.credentials
    RequestHelper.get(credentials.endpoint + crawlEntities.url, ctx)
}).parseResponse({ crawlEntitiesRequest, response, parser, ctx ->
    String lastUpdateCaptured
    parser.readStringValueAndThen(".QueryResult.Results[].LastUpdateDate", { lastUpdateCaptured = it })
    parser.readStringValueAndThen(".QueryResult.Results[]._ref", {

        def entity = new CrawlEntity()
        entity.url = it
        ctx.emit(entity)
    })
    parser.onFinish({
        def value = new NewHaveSeenContextValue()
        value.haveSeen = lastUpdateCaptured
        value.contextKeyToUse = crawlEntitiesRequest.contextKeyToUse
        ctx.emit(value)
        if (!crawlEntitiesRequest.contextKeyToUse.equals("seen")) {
            def incrementalValue = new NewHaveSeenContextValue()
            incrementalValue.haveSeen = lastUpdateCaptured
            incrementalValue.contextKeyToUse = "seen"
            ctx.emit(incrementalValue)
        }
        if (!lastUpdateCaptured.equals(crawlEntitiesRequest.previousLastUpdateDate)) {
            ctx.emit(new DecideToNack())
        }
    })
})

class DecideToNack {
}

PluginInterface.ackOrNack(DecideToNack).via({stream, ackOrNackDecision ->
    stream.get({
        ackOrNackDecision.nack()
    })
    null
})

PluginInterface.pushMessages(CrawlEntity).mapRequest({ crawlEntity ->
    def message = new CrawlEntityMessage()
    message.url = crawlEntity.url
    message.setMessageKey("2-crawlEntity-" + message.url)
    new PushCrawlerUMMessagesRequestVM()
        .setMessages([message])
})

PluginInterface.sendContextKeyValues(NewHaveSeenContextValue)
        .mapRequest({ value ->
    new SetContextKeyRequest()
            .setKey(value.contextKeyToUse)
            .setValue(value.haveSeen)
})

static int calculatePageSize(WaveContext waveContext) {
    List<String> filterValues = waveContext.crawlerMessageWithCredentials.credentials.getFilterValues("PAGE_SIZE");
    int pageSize
    if (filterValues.isEmpty()) {
        pageSize = 20
    } else {
        pageSize = Integer.parseInt(filterValues.get(0));
    }
    if (pageSize <= 1) {
        throw new IllegalArgumentException("Page size should be more then 1");
    }
    return pageSize;
}