import akka.http.javadsl.model.HttpRequest
import akka.http.javadsl.model.headers.HttpCredentials
import com.xebialabs.agatha.jsonstreaming.Parser
import com.xebialabs.impact.wave.scripting.PluginInterface
import com.xebialabs.impact.wave.scripting.http.WaveContext
import com.xebialabs.impact.wave.scripting.http.signals.Finish

import java.util.regex.Matcher
import java.util.regex.Pattern

class CrawledEntity {
    Long id
    String projectId
    String formattedId
    String objectType
    String version
    String refObjectUUID
    String creationDate
    String workspace
    String priority
    Map<String, Object> fieldValues = new HashMap<>()
    Map<String, Object> customFields = new HashMap<>()

    void addField(Parser parser, String fieldName) {
        parser.readStringValueAndThen("." + fieldName + "._refObjectName", { fieldValues.put(fieldName, it) })
    }

    def copyInto(CrawledEntity copy) {
        copy.id = id
        copy.formattedId = formattedId
        copy.objectType = objectType
        copy.version = version
        copy.refObjectUUID = refObjectUUID
        copy.creationDate = creationDate
        copy.workspace = workspace
        copy.fieldValues = new HashMap<>(fieldValues)
        copy.customFields = new HashMap<>(customFields)
        copy.projectId = projectId
        copy.priority = priority
    }
}

class TheUrl {
    String url

    TheUrl setUrl(String url) {
        this.url = url
        return this
    }
}

class CrawlRevisionHistory extends TheUrl {
    String workspaceId;
    String objectType
    CrawledFields crawledFields

    CrawlRevisionHistory setCrawledFields(CrawledFields crawledFields) {
        this.crawledFields = crawledFields
        return this
    }
}

def simplyGet = { TheUrl url, WaveContext ctx ->
    def credentials = ctx.crawlerMessageWithCredentials.credentials
    def auth = credentials.credentials[0]
    HttpRequest.GET(url.url)
            .addCredentials(HttpCredentials.createBasicHttpCredentials(
            auth.username,
            auth.password))
}

static boolean isCustomField(Parser parser, int level) {
    parser.getRelativePath(level).toString().startsWith(".c_");
}

static String getCustomFieldName(Parser parser, int level) {
    parser.getRelativePath(level).toString().substring(1);
}

static void addMultivalueFieldRequest(Parser parser, WaveContext ctx, String fieldName) {
    parser.readStringValueAndThen("." + fieldName + "._ref", {
        ctx.emit(new CrawlMultivalueFieldValue(fieldName).setUrl(it))
    })
}

class CrawlEntityWaitForFields {
    CrawlEntityMessage message

    CrawlEntityWaitForFields(CrawlEntityMessage message) {
        this.message = message
    }
}

PluginInterface.map(CrawlEntityMessage).into(CrawlEntityWaitForFields, RequestCustomFieldsFromCache).via({ message, ctx ->
    ctx.emit(new RequestCustomFieldsFromCache(false))
    ctx.emit(new CrawlEntityWaitForFields(message))
})

class CrawlEntityHavingCustomFields {
    CrawlEntityMessage message
    CrawledFields crawledFields
}

PluginInterface.reduce(CrawlEntityWaitForFields, CachedCustomFields).into(CrawlEntityHavingCustomFields).via({ stream ->
    CrawlEntityHavingCustomFields crawlEntity = new CrawlEntityHavingCustomFields()
    stream.get({ crawlEntity.message = it.message })
    stream.get(CachedCustomFields, {
        crawlEntity.crawledFields = it.fields
    })
    stream.onFinish({
        if (crawlEntity.message != null) {
            stream.emit(crawlEntity)
        }
    })
})

PluginInterface.request(
        CrawlEntityHavingCustomFields
).into(
        CrawledEntity,
        CrawlRevisionHistory,
        CrawlMultivalueFieldValue,
        CrawlRelations
).mapToRequest({ message, ctx -> simplyGet(new TheUrl().setUrl(message.message.url), ctx) }).parseResponse({ message, response, parser, ctx ->
    CrawledEntity crawledEntity = new CrawledEntity()
    List<CrawlRevisionHistory> crawlRevisionHistories = new ArrayList<>()
    parser.onObject({ objectPath ->
        crawledEntity.objectType = objectPath.subSequence(1, objectPath.length())
        parser.readLongValueAndThen(".ObjectID", { crawledEntity.id = it });
        parser.readStringValueAndThen(".LastUpdateDate", { crawledEntity.version = it });
        parser.readStringValueAndThen("._refObjectUUID", { crawledEntity.refObjectUUID = it });
        parser.readStringValueAndThen("._refObjectName", { crawledEntity.fieldValues.put("title", it) });
        parser.readStringValueAndThen(".CreationDate", { crawledEntity.creationDate = it });
        parser.readStringValueAndThen(".Priority", { crawledEntity.priority = it })
        parser.readStringValueAndThen(".RevisionHistory._ref", {
            crawlRevisionHistories.add((CrawlRevisionHistory) new CrawlRevisionHistory().setCrawledFields(message.crawledFields).setUrl(it))
        });
        parser.readStringValueAndThen(".State", {
            if (it != null && !it.isEmpty()) {
                crawledEntity.fieldValues.put("state", it)
            }
        });
        parser.readStringValueAndThen(".ScheduleState", {
            if (it != null && !it.isEmpty()) {
                crawledEntity.fieldValues.put("state", it)
            }
        })

        crawledEntity.addField(parser, "Owner");
        crawledEntity.addField(parser, "Project");
        parser.readStringValueAndThen(".Project._ref", { ref ->
            int i = ref.lastIndexOf('/')
            if (i != -1) {
                crawledEntity.projectId = ref.substring(i + 1)
            }
        })
        crawledEntity.addField(parser, "SubmittedBy");
        crawledEntity.addField(parser, "Subscription");
        crawledEntity.addField(parser, "Workspace");
        crawledEntity.addField(parser, "FlowState");
        parser.readStringValueAndThen(".FormattedID", {crawledEntity.formattedId = it})
        parser.readStringValueAndThen(".Workspace._ref", {
            int pos = it.lastIndexOf('/')
            crawledEntity.workspace = it.substring(pos + 1)
        })
        int level = parser.getCurrentLevel();

        // custom fields
        parser.onValue({ path, type ->
            if (isCustomField(parser, level)) {
                // this is custom field
                parser.readDataTillEofAndThen({ value ->
                    String customFieldName = getCustomFieldName(parser, level);
                    crawledEntity.customFields.put(customFieldName, value);
                });
            }
        });

        parser.onObject({ path ->
            if (isCustomField(parser, level)) {
                String customFieldName = getCustomFieldName(parser, level);
                List<String> values = new ArrayList<>();
                parser.readStringValueAndThen("._tagsNameArray[].Name", { values.add(it) });
                parser.onFinish({ crawledEntity.customFields.put(customFieldName, values) });
            }
        });

        addMultivalueFieldRequest(parser, ctx, "Milestones")
        addMultivalueFieldRequest(parser, ctx, "Tags")


        addLinkUrl(parser, ctx, "Connections");
        addLinkUrl(parser, ctx, "Children");
        addLinkUrl(parser, ctx, "Defects");
        addLinkUrl(parser, ctx, "DefectSuites");
        addLinkUrl(parser, ctx, "Duplicates");
        addLinkUrl(parser, ctx, "Predecessors");
        addLinkUrl(parser, ctx, "Results");
        addLinkUrl(parser, ctx, "Risks");
        addLinkUrl(parser, ctx, "Steps");
        addLinkUrl(parser, ctx, "Successors");
        addLinkUrl(parser, ctx, "Tasks");
        addLinkUrl(parser, ctx, "TestCases");

    })
    parser.onFinish({
        ctx.emit(crawledEntity)
        crawlRevisionHistories.forEach({
            it.workspaceId = crawledEntity.workspace
            it.objectType = crawledEntity.objectType
            ctx.emit(it)
        })
    })
})

class CrawlMultivalueFieldValue extends TheUrl {
    String fieldName

    CrawlMultivalueFieldValue(String fieldName) {
        this.fieldName = fieldName
    }
}

class CrawledMultivalueFieldValue {
    String fieldName
    List<String> values = new ArrayList<>()
}

PluginInterface.request(CrawlMultivalueFieldValue).into(CrawledMultivalueFieldValue).mapToRequest(simplyGet).parseResponse({ o, response, parser, ctx ->
    CrawledMultivalueFieldValue crawledMultivalueFieldValue = new CrawledMultivalueFieldValue()
    crawledMultivalueFieldValue.fieldName = o.fieldName
    parser.onObject(".QueryResult.Results[]", {
        StringBuilder name = parser.readValueInto(".Name")
        StringBuilder formattedId = parser.readValueInto(".FormattedID")
        parser.onFinish({
            List<String> pc = new ArrayList<>()
            if (formattedId.length() > 0) {
                pc.add(formattedId.toString())
            }
            if (name.length() > 0) {
                pc.add(name.toString())
            }
            crawledMultivalueFieldValue.values.add(pc.join(": "))
        })
    })
    parser.onFinish({ ctx.emit(crawledMultivalueFieldValue) })
})

class CrawlRevisions extends TheUrl {
    String workspaceId
    String objectType
    CrawledFields crawledFields
}

PluginInterface.request(CrawlRevisionHistory).into(CrawlRevisions).mapToRequest(simplyGet).parseResponse({ crawlRevisionHistory, response, parser, ctx ->
    parser.readStringValueAndThen(".RevisionHistory.Revisions._ref", {
        def revisions = new CrawlRevisions()
        revisions.objectType = crawlRevisionHistory.objectType
        revisions.workspaceId = crawlRevisionHistory.workspaceId
        revisions.crawledFields = crawlRevisionHistory.crawledFields
        ctx.emit(revisions.setUrl(it))
    })
})

class CrawledRevision {
    String description
    String createdAt
    int revisionNumber
    String workspaceId
    String objectType
    CrawledFields crawledFields
}

PluginInterface.request(CrawlRevisions).into(CrawledRevision).mapToRequest(simplyGet).parseResponse({ crawlRevisions, respose, parser, ctx ->
    parser.onObject(".QueryResult.Results[]", {
        def crawledRevision = new CrawledRevision()
        crawledRevision.crawledFields = crawlRevisions.crawledFields
        crawledRevision.workspaceId = crawlRevisions.workspaceId
        crawledRevision.objectType = crawlRevisions.objectType
        parser.readStringValueAndThen(".Description", { crawledRevision.description = it })
        parser.readStringValueAndThen(".CreationDate", { crawledRevision.createdAt = it })
        parser.readIntValueAndThen(".RevisionNumber", { crawledRevision.revisionNumber = it })
        parser.onFinish({ ctx.emit(crawledRevision) })
    })
})

class ParsedRevisionDescription {
    List<HistoryChange> changes = new ArrayList<>()
    CrawledRevision revision
}

class HistoryChange {
    String raw
    String field
    String oldData
    String newData
    String removed
    String added
    int index
    String customFieldId
    FieldType customFieldType
    String normalFieldId
    FieldType normalFieldType
}


PluginInterface.map(CrawledRevision).into(HistoryChange).via({ crawledRevision, ctx ->
    String ADDED = "added";
    String CHANGED_FROM = "changed from";
    String REMOVED = "removed";

    String REGEXP = '(^|, )([^a-z]+)\\s+(' + ADDED + ' \\[(.*?)\\]|' + CHANGED_FROM + ' \\[(.*?)\\] to \\[(.*?)\\]|' + REMOVED + ' \\[(.*?)\\])($|, )';

    String description = crawledRevision.description
    Matcher matcher = Pattern.compile(REGEXP).matcher(description);
    int index = 0
    int start = 0

    println("pre")
    while (matcher.find(start)) {
        println("in")
        HistoryChange change = new HistoryChange()
        change.field = matcher.group(2)
        String customFieldId = crawledRevision.workspaceId + '/' + crawledRevision.objectType + '/' + change.field.toUpperCase()
        def matchingCustomField = crawledRevision.crawledFields.fieldsByDisplayName.get(customFieldId)
        if (matchingCustomField != null) {
            change.customFieldId = matchingCustomField.elementName
            change.customFieldType = matchingCustomField.fieldType
        } else {
            if (change.field.equals("SCHEDULE STATE")) {
                change.normalFieldId = "state"
                change.normalFieldType = FieldType.STRING
            }
        }
        change.index = index
        change.raw = description
        start = matcher.start(8);
        String action = matcher.group(3);
        if (action.startsWith(ADDED)) {
            change.added = matcher.group(4)
        } else if (action.startsWith(CHANGED_FROM)) {
            change.oldData = matcher.group(5)
            change.newData = matcher.group(6)
        } else if (action.startsWith(REMOVED)) {
            change.removed = matcher.group(7)
        } else {
            throw new IllegalStateException("Unknown action: [" + action + "]");
        }
        println("preemit")
        if (change.customFieldId != null || change.normalFieldId != null) {
            ctx.emit(change)
        }
        println("postemit")
        println(index)
        if (index++ > 1000) {
            throw new IllegalStateException("too many changes")
        }
    }
})

PluginInterface.reduce(HistoryChange).by(CrawledRevision, { it }).into(ParsedRevisionDescription).via({ stream ->
    ParsedRevisionDescription result = new ParsedRevisionDescription()
    result.revision = (CrawledRevision) stream.getBucket().get(0)
    stream.get({
//        stream.emit(new LogMessage(LogMessage.Level.INFO, "got something"))
        result.changes.add(it)
    })
    stream.onFinish({
//        stream.emit(new LogMessage(LogMessage.Level.INFO, "finished"))
        stream.emit(result)
    })
})

class CrawledEntityCurrentVersion {
    CrawledEntity entity
    Map<String, List<String>> relations = new HashMap<>()
}

PluginInterface.reduce(
        CrawledEntity,
        CrawledMultivalueFieldValue,
        CrawledRelations
).into(
        CrawledEntityCurrentVersion
).via({ stream ->
    CrawledEntityCurrentVersion crawledEntityCurrentVersion = new CrawledEntityCurrentVersion()
    List<CrawledMultivalueFieldValue> fields = new ArrayList<>()
    stream.get({
        crawledEntityCurrentVersion.entity = it
    })
    stream.get(CrawledMultivalueFieldValue, {
        fields.add(it)
    })
    stream.get(CrawledRelations, {
        crawledEntityCurrentVersion.relations.put(it.name, it.relatedIds)
    })
    stream.onFinish({
        fields.each({
            crawledEntityCurrentVersion.entity.fieldValues.put(it.fieldName, it.values)
        })
        stream.emit(crawledEntityCurrentVersion)
    })
})

PluginInterface.reduce(
        CrawledEntityCurrentVersion,
        ParsedRevisionDescription
).into(
        CrawledEntityRevision
).via({ stream ->
    println("asdfasdf")
    List<ParsedRevisionDescription> changes = new ArrayList<>()
    CrawledEntityCurrentVersion currentVersion
    stream.get(ParsedRevisionDescription, {
        changes.add(it)
    })
    stream.get({
        currentVersion = it
    })
    stream.onFinish({
        CrawledEntityRevision pastRevision = new CrawledEntityRevision(currentVersion)
        pastRevision.isCurrentRevision = true
        pastRevision.entity.version = changes.size()
        stream.emit(pastRevision)
        changes.sort(Comparator.comparing({it.revision.revisionNumber}))
        for (int i = changes.size() - 1; i >= 0; i --) {
            pastRevision = new CrawledEntityRevision(pastRevision)
            pastRevision.isCurrentRevision = false
            pastRevision.entity.version = i
            pastRevision.entity.creationDate = changes.get(i).revision.createdAt
            changes.get(i).changes.forEach({change ->
                if (change.normalFieldId != null) {
                    applyChange(change, change.normalFieldId, change.normalFieldType, pastRevision.entity.fieldValues)
                } else if (change.customFieldId != null) {
                    applyChange(change, change.customFieldId, change.customFieldType, pastRevision.entity.customFields)
                }
            })
            stream.emit(pastRevision)
        }
    })
})

static def applyChange(HistoryChange change, String fieldName, FieldType fieldType, Map<String, Object> fields) {
    if (fieldType == FieldType.MULTIVALUE) {
        List<String> values = (List<String>)fields.get(fieldName)
        values = values == null ? new ArrayList<>() : new ArrayList<>(values)
        if (change.removed != null) {
            values.add(change.removed)
        }
        if (change.added != null) {
            values.remove(change.added)
        }
        fields.put(fieldName, values)
    } else if (fieldType == FieldType.STRING) {
        if (change.added != null) {
            fields.put(fieldName, "")
        } else if (change.removed != null) {
            fields.put(fieldName, change.removed)
        } else if (change.oldData != null) {
            fields.put(fieldName, change.oldData)
        }
    }
}

class CrawledEntityRevision extends CrawledEntityCurrentVersion {
    boolean isCurrentRevision

    CrawledEntityRevision() {
    }

    CrawledEntityRevision(CrawledEntityCurrentVersion copy) {
        relations = copy.relations
        this.entity = new CrawledEntity()
        copy.entity.copyInto(this.entity)
    }


}

class CrawlRelations extends TheUrl {
    String name

    CrawlRelations() {
    }

    CrawlRelations(String name) {
        this.name = name
    }
}

def addLinkUrl(Parser parser, WaveContext ctx, String name) {
    parser.readStringValueAndThen("." + name + "._ref", { ref ->
        ctx.emit(new CrawlRelations(name).setUrl(ref))
    });
}

class CrawledRelations {
    String name
    List<String> relatedIds = new ArrayList<>()
}

PluginInterface.request(
        CrawlRelations
).into(
        CrawledRelations
).mapToRequest(simplyGet).parseResponse({crawlRelations, response, parser, ctx ->
    CrawledRelations crawledRelations = new CrawledRelations()
    crawledRelations.name = crawlRelations.name
    parser.readStringValueAndThen(".QueryResult.Results[]._ref", { ref ->
        int pos = ref.lastIndexOf("/");
        if (pos != -1) {
            crawledRelations.relatedIds.add(ref.substring(pos + 1));
        }
    })
    parser.onFinish({ctx.emit(crawledRelations)})
})


