import sys

from compare.modules.propertyUtil import PropertyUtil
from com.xebialabs.xldeploy.extensions import ComparisonTopologyMappingParser as ruleparser


class DiscoveryResultComparator(object):
    def __init__(self, metadata_service, inspection_service, repository_service, task_ids):
        self._metadata_service = metadata_service
        self._repository_service = repository_service
        self._inspection_service = inspection_service
        self._task_ids = task_ids

        self._root_cis = []
        self._simple_kinds = ['BOOLEAN', 'INTEGER', 'STRING', 'ENUM', 'DATE', 'SET_OF_STRING', 'LIST_OF_STRING', 'MAP_STRING_STRING']
        self._root_discovered_cis_tuples = []
        self._ids_cis_map = {}
        self._rules = ruleparser.parsedMappings()

    def _get_root_id(self, index):
        return self._root_cis[index].id

    def _populate_ids_cis_map(self, cis_list):
        for each_ci in cis_list:
            self._ids_cis_map[each_ci.id] = each_ci

    def compare_discovery_tasks_results(self):
        # array of arrays
        inspected_cis = []
        for task_id in self._task_ids:
            sorted_cis_list = self._sort_results(self._inspection_service.retrieveInspectionResults(task_id))
            self._populate_ids_cis_map(sorted_cis_list)
            inspected_cis.append(sorted_cis_list)
            root_ci = sorted_cis_list[0]
            self._root_cis.append(root_ci)

        if len(self._task_ids) == 1:
            self._root_cis.insert(0, self._repository_service.read(self._root_cis[0].id))
            repo_ci_ids = self._repository_service.query(None, None, self._root_cis[0].id, None, None, None, 0, -1)
            repo_cis = map(lambda ci_id: self._repository_service.read(ci_id.id), repo_ci_ids)
            repo_cis.append(self._repository_service.read(self._root_cis[0].id))
            inspected_cis.insert(0, self._sort_results(repo_cis))
            self._populate_ids_cis_map(repo_cis)

        lined_up_comparable_cis = self._line_up_cis(inspected_cis)
        return self._compare_cis_with_props(lined_up_comparable_cis)

    def _sort_results(self, cis_list):
        return sorted(cis_list, key=lambda ci: ci.id.count('/'))

    def _print_cis_list(self, desc, ci_list):
        ids = []
        for each in ci_list:
            if each is None:
                ids.append('None')
            else:
                ids.append(each.id)
        logger.info(desc + ": [" + ", ".join(ids) + "]")

    # Returns a list of lined-up comparable CIs
    # e.g. [[node-dev, node-test, node-acc], [server1-dev, None, server1-acc], [None, server2-test, server2-acc]]
    # It loops through each discovered CIs list and tries to find a matching CI from the other list.
    # If it succeeds in finding one, it removes the item from that list.
    #
    # assuming a list of discovered cis [[a, b, c], [a', b'], [a'', c'']]
    # it proceeds in the following manner:
    # 1st pass =[[b, c], [b'], [c'']] lined-up-cis = [[a,a',a'']]
    # 2nd pass =[[c], [], [c'']] lined-up-cis = [[a,a',a''], [b, b', None]]
    # 3rd pass =[[], [], []] lined-up-cis = [[a,a',a''], [b, b', None], [c, None, c'']
    def _line_up_cis(self, cis_list):
        similar_cis_groups_list = []
        for count in range(0, len(cis_list)):
            curr_cis_list = cis_list[count]
            # self.print_cis_list('curr_cis_list', curr_cis_list)

            while len(curr_cis_list) > 0:
                curr_ci = curr_cis_list[0]
                #print 'curr_ci %s' % (curr_ci.name)
                line_up_cis = []
                # collect equivalent cis from other list
                for list_counter in range(0, len(cis_list)):
                    searched_cis_list = cis_list[list_counter]
                    #self.print_cis_list('searched_cis_list ' + str(list_counter), searched_cis_list)
                    if len(searched_cis_list) == 0:
                        line_up_cis.insert(list_counter, None)

                    elif list_counter == count:
                        line_up_cis.insert(list_counter, curr_ci)
                        curr_cis_list.remove(curr_ci)

                    else:
                        equivalent_ci = self._find_equivalent_from_list(curr_ci, searched_cis_list)
                        if equivalent_ci is not None:
                            line_up_cis.insert(list_counter, equivalent_ci)
                            cis_list[list_counter].remove(equivalent_ci)
                        else:
                            line_up_cis.insert(list_counter, None)
                # self._print_cis_list('comparable cis created', line_up_cis)
                similar_cis_groups_list.append(line_up_cis)

        return similar_cis_groups_list


    def _find_equivalent_from_list(self, ci, searched_cis_list):
        for each in searched_cis_list:
            if each is not None and self._are_cis_equivalent(ci, each):
                return each
        return None

    def _are_cis_equivalent(self, ci1, ci2):

        if ci1.type.toString() != ci2.type.toString():
            return False
        elif (ci1 in self._root_cis) and (ci2 in self._root_cis):
            return True
        elif ci1.name == ci2.name:
            return self._are_cis_equivalent(self._find_parent(ci1.id), self._find_parent(ci2.id))
        elif ci1.type.toString() in self._rules:
            rule = self._rules[ci1.type.toString()]
            return self._ci_equivalent_by_rule(ci1, ci2, rule)

    def _ci_equivalent_by_rule(self, lhs, rhs, rule):
        # TODO: 'eval' is temporary and unsafe way. Find out a better way to evaluate expression on a ci

        ruleResult = False
        try:
            ruleResult = eval(rule)
        except:
            exc_type, exc_obj, exc_tb = sys.exc_info()
            logger.warn('comparison failed between %s and %s: %s' % (lhs.id, rhs.id, exc_obj))

        return ruleResult and self._are_cis_equivalent(self._find_parent(lhs.id), self._find_parent(rhs.id))

    def _find_parent(self, ci_id):
        parent_id = ci_id[0:ci_id.rindex('/')]
        return self._ids_cis_map[parent_id]

    def _compare_cis_with_props(self, similar_cis_groups_list):
        results = {"/": self._compare_roots()}

        # put the root cis properties comparison results under '/'

        # other non-root cis comparisons
        del similar_cis_groups_list[0]

        for each_comparable_group in similar_cis_groups_list:
            rootless_id = self._find_rootless_id(each_comparable_group)
            results[rootless_id] = {}
            type_descriptor = self._metadata_service.findDescriptor(self._find_type(each_comparable_group))

            compared_props = self._compare_cis_props(each_comparable_group, type_descriptor)

            results[rootless_id] = self._set_id_type_info_for_comparison(results[rootless_id], each_comparable_group, type_descriptor)

            len_compared_cis = len([ci for ci in each_comparable_group if ci is not None])
            if len_compared_cis == 1 or compared_props == []:
                results[rootless_id]["properties"] = None
            else:
                results[rootless_id]["properties"] = compared_props
        return results

    def _set_id_type_info_for_comparison(self, result, comparable_cis, type_descriptor):
        result["ci_type"] = type_descriptor.type.toString()
        result["ci_paths"] = []
        for counter in range(0, len(comparable_cis)):
            root_ci_id = self._root_cis[counter].id

            if comparable_cis[counter] is not None:
                ci_id = comparable_cis[counter].id
                if ci_id != root_ci_id:
                    ci_id = ci_id.replace(root_ci_id, "")
                if ci_id.startswith('/'):
                    ci_id = ci_id[1:]
                result["ci_paths"].append(ci_id)
            else:
                result["ci_paths"].append(None)
        return result

    # collects root CIs from discovery results and compare them
    def _compare_roots(self):
        result = {"properties": None}
        type_descriptor = self._metadata_service.findDescriptor(self._root_cis[0].type)
        result = self._set_id_type_info_for_comparison(result, self._root_cis, type_descriptor)

        compared_props = self._compare_cis_props(self._root_cis, type_descriptor)
        if compared_props != {}:
            result["properties"] = compared_props

        return result

    def _find_rootless_id(self, each_comparable_cis_list):
        for index in range(0, len(each_comparable_cis_list)):
            each = each_comparable_cis_list[index]
            if each is not None:
                id = each.id
                corresponding_root_id = self._root_cis[index].id
                return id.replace(corresponding_root_id + "/", "")
        return None

    def _find_type(self, each_comparable_cis_list):
        for each in each_comparable_cis_list:
            if each is not None:
                return each.type
        return None

    def _compare_cis_props(self, cis, type_descriptor):
        property_descriptors = [prop_desc for prop_desc in type_descriptor.getPropertyDescriptors()]

        # e.g. [{'prop1':['value1', 'value2', 'value3']}, {'prop2':['value1', 'value2', 'value3']...}
        ci_prop_values = []

        for prop in property_descriptors:
            kind = prop.getKind().name()
            if not prop.isHidden() and kind in self._simple_kinds:
                prop_values_map = {prop.name: []}

                for ci_index in range(0, len(cis)):
                    ci = cis[ci_index]
                    if ci is not None:
                        string_value = PropertyUtil.to_string(prop.get(ci), kind)
                        prop_values_map[prop.name].append(string_value)

                # if all the values of the property are same, don't include it in the result
                if len(prop_values_map[prop.name]) > 1 and not all(prop_values_map[prop.name][0] == prop_value for prop_value in prop_values_map[prop.name]):
                    ci_prop_values.append(prop_values_map)

        return ci_prop_values
