package com.xebialabs.xltest.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.TestRunId;
import com.xebialabs.xltest.domain.EventHandler;
import com.xebialabs.xltest.repository.EventNotifier;

/**
 * Fixme: we're going to run into some nasty concurrency issues here.
 */
public abstract class EventDispatcher implements EventRegistry, EventNotifier {
    private final static Logger LOG = LoggerFactory.getLogger(EventDispatcher.class);

    private final Multimap<TestRunId, EventHandler> runSpecificHandlers;

    public EventDispatcher() {
    	this.runSpecificHandlers = Multimaps.synchronizedMultimap(LinkedListMultimap.<TestRunId, EventHandler>create());
    }

    @Override
    public void notify(TestRunId testRunId, Event event) {
        List<EventHandler> runSpecificEventHandlers = getRunSpecificEventHandlers(testRunId);
        LOG.info("Invoking "+ runSpecificEventHandlers.size() +" handlers for event of type {}", event.getType());
        for (EventHandler handler : runSpecificEventHandlers) {
            invokeEventHandler(testRunId, handler, event);
        }
    }

	public List<EventHandler> getRunSpecificEventHandlers(TestRunId testRunId) {
        ArrayList<EventHandler> eventHandlers = new ArrayList<EventHandler>(16);
        for (TestRunId id = testRunId; ; id = id.getParent()) {
            Collection<EventHandler> h = runSpecificHandlers.get(id);
            if (h != null) {
                eventHandlers.addAll(0, h);
            }

            // Quit loop after root elements are added to the handlers list.
            if (id.isRoot()) {
                break;
            }
        }
        return eventHandlers;
    }

    protected abstract void invokeEventHandler(TestRunId testRunId, EventHandler handler, Event event);

    public void registerHandler(EventHandler handler) {
        registerHandler(TestRunId.ROOT, handler);
    }

    public void registerHandler(TestRunId runId, EventHandler handler) {
        runSpecificHandlers.put(runId, handler);
    }

    public void unregisterAll(TestRunId runId) {
        for (Iterator<TestRunId> iter = runSpecificHandlers.keySet().iterator(); iter.hasNext(); ) {
            if (iter.next().isSameOrChildOf(runId)) {
                iter.remove();
            }
        }
    }
    
}
