package com.xebialabs.xltest.resources;

import static com.xebialabs.deployit.plugin.api.reflect.Type.valueOf;

import java.util.Collection;

import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;

import org.eclipse.jetty.server.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.xltest.domain.EventProcessor;
import com.xebialabs.xltest.domain.Report;
import com.xebialabs.xltest.domain.Store;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.domain.TestSetDefinition;

@Controller
@Path("/data")
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
public class DataController {
	
	private static final Logger LOG = LoggerFactory.getLogger(DataController.class.getName());

    private RepositoryService repository;

    @Autowired
    public DataController(RepositoryService repository) {
        this.repository = repository;
    }

    @GET
    @Path("/")
    public Collection<BaseConfigurationItem> getAll(@QueryParam("type") String type)  {
        SearchParameters query = new SearchParameters();
        if (type != null) {
            query.setType(valueOf(type));
        }
        return repository.listEntities(query);
    }

    @GET
    @Path("/{id:.+}")
    public BaseConfigurationItem get(@PathParam("id") String id)  {
    	LOG.info("reading item with id: " + id);
        return repository.read(id);
    }

    @PUT
    @Path("/{id:.+}")
    public BaseConfigurationItem update(@PathParam("id") String id, BaseConfigurationItem item, @Context final HttpServletResponse response)  {
    	LOG.info("update item with id: " + item.getId());
        // Should rename item on id 'id' to item on id 'item.id'?
    	if (id.equals(item.getId())) {
    		for (PropertyDescriptor propertyDescriptor : item.getType().getDescriptor().getPropertyDescriptors()) {
    			String name = propertyDescriptor.getName();
    			Object propertyValue = propertyDescriptor.get(item);
    			LOG.info("set property " + name + " to value: " + propertyValue); 
    		}
            repository.update(item);
            response.setStatus(Response.SC_CREATED);
        } else {
            response.setStatus(Response.SC_CONFLICT);
        }

        return item;
    }

    @POST
    @Path("/")
    public BaseConfigurationItem create(BaseConfigurationItem item, @Context final HttpServletResponse response) {
        addPrefixIfNeeded(item);
        LOG.info("create item with id: " + item.getId());
    	repository.create(item);
        response.setStatus(Response.SC_CREATED);
        return item;
    }

    @DELETE
    @Path("/{id:.+}")
    @Consumes(MediaType.WILDCARD)
    public void delete(@PathParam("id") String id, @Context final HttpServletResponse response) {
        LOG.info("delete item with id: " + id);
        repository.delete(id);
        response.setStatus(Response.SC_OK);
    }

	public void addPrefixIfNeeded(BaseConfigurationItem item) {
		String id = item.getId();
		if (id.startsWith("Configuration") || id.startsWith("Applications")) {
			return;
		}
		String rootNodeName = item.getType().getDescriptor().getRoot().getRootNodeName();
		if (rootNodeName == null) {
			throw new RuntimeException("Type " + item.getType() + " is unrooted!");
		}
		String subdir = "";
		if (item.getType().instanceOf(Type.valueOf(Report.class))) {
			subdir = "Reports";
		}
		if (item.getType().instanceOf(Type.valueOf(Store.class))) {
			subdir = "Stores";
		}
		if (item.getType().instanceOf(Type.valueOf(EventProcessor.class))) {
			subdir = "Listeners";
		}
//		if (item.getType().instanceOf(Type.valueOf(Record.class))) {
//			subdir = "Records";
//		}
		if (item.getType().instanceOf(Type.valueOf(TestSetDefinition.class))) {
			subdir = "TestSetDefinitions";
		}
		if (item.getType().instanceOf(Type.valueOf(TestRun.class))) {
			subdir = "TestRuns";
		}
		item.setId(rootNodeName + "/" + subdir + "/" + id);
	}
}
