package com.xebialabs.xltest.repository;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.xltest.domain.FreemarkerReport;
import com.xebialabs.xltest.domain.Report;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.domain.TestRunId;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.AutowireCapableBeanFactory;
import org.springframework.stereotype.Repository;

import javax.ws.rs.core.UriBuilder;

import java.net.URI;
import java.util.*;

import static com.xebialabs.deployit.plugin.api.reflect.Type.valueOf;

@Repository
public class TestRuns {
	
	private static Logger LOG = LoggerFactory.getLogger(TestRuns.class);

    public static final String TEST_RUNS_ROOT = "Applications/TestRuns/";
    private final RepositoryService repository;
    private final AutowireCapableBeanFactory beanFactory;

    @Autowired
    public TestRuns(RepositoryService repository, AutowireCapableBeanFactory beanFactory) {
        this.repository = repository;
        this.beanFactory = beanFactory;
    }

	public TestRun addRunOfTestSet(TestRun testRun, URI absolutePath) {

        // What I want: create a RunTestSet command (event) and let test runners pick up the command
        // Now this works because the command is blocked as long as
        TestRunId newTestRunId = generateNewRunId();
        URI uri = getTestRunUrl(absolutePath, newTestRunId);

        // TODO: Build test runner config based on test set id and additional parameters.
        testRun.setId(TEST_RUNS_ROOT + newTestRunId.toString());
        testRun.setStartTime(startTime());
        testRun.setUri(uri);

        beanFactory.autowireBean(testRun);

        repository.create(testRun);

        return testRun;
    }

    public void updateTestRun(TestRun testRun) {
        repository.update(testRun);
    }

    public TestRun getTestRun(TestRunId testRunId) {
        TestRun testRun = repository.read(TEST_RUNS_ROOT + testRunId.toString());
        //beanFactory.autowireBean(testRun);
        return testRun;
    }

    private Date startTime() {
        return new Date();
    }

    private TestRunId generateNewRunId() {
        return new TestRunId(UUID.randomUUID().toString());
    }

    private URI getTestRunUrl(URI absolutePath, TestRunId testRunId) {
        return UriBuilder.fromUri(absolutePath).path(testRunId.toString()).build();
    }

    /**
     * Note: Items are not auto-wired.
     *
     * @return
     */
    public Collection<TestRun> getAllTestRuns() {
        return getTestRuns(new SearchParameters());
    }

    public Collection<TestRun> getTestRuns(SearchParameters query) {
        query.setType(valueOf(TestRun.class));
        List<TestRun> testRuns = repository.listEntities(query);
        Collections.sort(testRuns, new Comparator<TestRun>() {
            @Override
            public int compare(TestRun o1, TestRun o2) {
                if (o1.getStartTime() == null) {
                    return 1;
                }
                if (o2.getStartTime() == null) {
                    return -1;
                }
                return o2.getStartTime().compareTo(o1.getStartTime());
            }
        });
        return testRuns;
    }

    public Map<String, Object> getQualification(TestRunId testRunId) {
        TestRun run = getTestRun(testRunId);
        Report qualification = run.getTestSetDefinition().getQualification();
        return generateReport(run, qualification, Collections.<String, String>emptyMap());
    }

    public Map<String, Object> generateReport(TestRunId testRunId, Report report, Map<String, String> queryParameters) {
        TestRun run = getTestRun(testRunId);
        return generateReport(run, report, queryParameters);
    }

    private Map<String, Object> generateReport(TestRun run, Report report, Map<String, String> queryParameters) {
        if (report != null) {
            beanFactory.autowireBean(report);
            report.addAttribute("queryParameters", queryParameters);

            Map<String, Object> resultObject = new TreeMap();
            resultObject.put("type", report.getReportType().toLowerCase());
            resultObject.put("report", report.compute(run));
            return resultObject;
        }
        return Collections.emptyMap();
    }
}
