package com.xebialabs.xltest.utils;

import org.apache.http.Header;
import org.apache.http.HttpMessage;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.*;
import org.apache.http.client.params.ClientPNames;
import org.apache.http.client.params.CookiePolicy;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Base Fixture class.
 */
public class JsonOverHttpHelper {
	
	private static final Logger LOG = LoggerFactory.getLogger(JsonOverHttpHelper.class.getName());

	private HttpClient httpClient;
	private final HttpContext localContext;

	private ResponseParser responseParser;
    private String contentType;

	private HttpResponse response;
	private String content;
	
	public JsonOverHttpHelper() {
		httpClient = new DefaultHttpClient();
		httpClient.getParams().setParameter(
		        ClientPNames.COOKIE_POLICY, CookiePolicy.BROWSER_COMPATIBILITY);
		
		localContext = new BasicHttpContext();

		expectOutput("json");
	}
	
	public void expectOutput(final String format) {
		if ("json".equalsIgnoreCase(format)) {
			responseParser = new JsonResponseParser();
		}
	}
	
	public void httpGetRequest(final String url) throws Exception {
		LOG.info("doing a get on: " + url);
		HttpGet request = new HttpGet(url);
		addDefaultHeaders(request);
		executeRequest(request);
	}
	
	public void httpDeleteRequest(final String url) throws Exception {
		LOG.info("doing a delete on: " + url);
		HttpDelete request = new HttpDelete(url);
		addDefaultHeaders(request);
		executeRequest(request);
	}

    private void addDefaultHeaders(final HttpMessage httpMessage) {
        httpMessage.setHeader("Accept", responseParser.acceptedMimeType());
    }

	public void httpPostRequest(final String url, final String content) throws Exception {
		LOG.info("doing a post on: " + url + " with content: " + content);
		HttpPost request = new HttpPost(url);
        StringEntity entity = new StringEntity(content, getContentTypeForSubmittedData(), "UTF-8");
		request.setEntity(entity);
        addDefaultHeaders(request);
		executeRequest(request);
	}

    public void httpPutRequest(final String url, final String content) throws Exception {
		LOG.info("doing a put on: " + url + " with content: " + content);
		HttpPut request = new HttpPut(url);
        StringEntity entity = new StringEntity(content, getContentTypeForSubmittedData(), "UTF-8");
		request.setEntity(entity);
        addDefaultHeaders(request);
		executeRequest(request);
	}

    private String getContentTypeForSubmittedData() {
        return contentType != null ? contentType : responseParser.acceptedMimeType();
    }

	private void executeRequest(final HttpUriRequest request) throws Exception {
		response = httpClient.execute(request, localContext);
		content = EntityUtils.toString(response.getEntity());
		responseParser.parse(content);
	}

	public int statusCode() {
		assertResponse();
		return response.getStatusLine().getStatusCode();
	}

	public String contentType() {
		return header("Content-Type");
	}

	public String header(final String header) {
		assertResponse();
		Header headerValue = response.getFirstHeader(header);
		return headerValue != null ? headerValue.getValue() : null;
	}

	public String content() {
		assertResponse();
		return content;
	}
	
	public boolean contentContains(final String substring) {
		assertResponse();
		return content.contains(substring);
	}

	public Object path(final String path) {
		assertResponse();
		Object valueOnPath = responseParser.getValue(path);
		LOG.info(responseParser.getClass().getName() + " got value: " + valueOnPath + " on path: " + path);
		return valueOnPath;
	}
	
	// Object will be either Integer or String
	public int pathCount(final String path) {
	    assertResponse();
	    return responseParser.getCount(path);
	}
	
	private void assertResponse() {
		if (response == null) {
			throw new AssertionError("First perform a Http GET or POST request");
		}
	}

    public void setContentType(String contentType) {
        this.contentType = contentType;
    }

	// For testing
	void setHttpClient(final HttpClient httpClient) {
		this.httpClient = httpClient;
	}
}
