package com.xebialabs.xltest.jenkins;

import com.google.common.base.Charsets;
import com.google.common.base.Strings;
import com.google.common.io.CharStreams;
import com.google.common.net.UrlEscapers;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.spi.JsonProvider;
import com.jayway.jsonpath.spi.JsonProviderFactory;
import net.minidev.json.JSONArray;
import net.minidev.json.JSONObject;
import org.jboss.resteasy.util.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Sends events to XL Test with information found in the wiki pages.
 */
public class JenkinsQueueInspector {
    private static final Logger LOG = LoggerFactory.getLogger(JenkinsQueueInspector.class.getName());

    public static final String SUCCESS = "SUCCESS";
    public static final int MINUTE = 60 * 1000;
    public static final int ONE_HOUR = 60 * MINUTE;
    public static final String XLTEST_URL = "XLTEST_URL";

    /* Exit codes */
    public static final int BUILD_SUCCESS = 0;
    public static final int BUILD_FAILURE = 1; // failure + unstable
    public static final int BUILD_TIMEOUT = 2;

    static final JsonPath BUILD_XLTEST_RESULT = JsonPath.compile("$.result");
    private final String username;
    private final String password;
    private final String uniqueIdParameterName;
    private final String uniqueId;
    private final String jobName;

    // the jenkins base url is something like: "http://localhost:8080"
    private URL jenkinsBaseUrl;
    private Number buildNumber;
    private String slaveName;

    public JenkinsQueueInspector(URL url, String jobName, String uniqueId) {

        this.jenkinsBaseUrl = url;
        username = password = null;
        uniqueIdParameterName = XLTEST_URL;
        this.uniqueId = uniqueId;
        this.jobName = jobName;
    }

    public JenkinsQueueInspector(URL url, String jobName, String username, String password, String uniqueIdParameterName, String uniqueId) {
        this.jenkinsBaseUrl = url;
        this.jobName = jobName;
        this.username = username;
        this.password = password;
        this.uniqueIdParameterName = uniqueIdParameterName;
        this.uniqueId = uniqueId;
    }

    public int inspectNumberOfJobsInBuildQueue() throws Exception {
        StatusCodeAndContent statusCodeAndContent = null;
        int retries = 1200; // keep retrying for max 10 min
        while (statusCodeAndContent != null && statusCodeAndContent.statusCode != 200 && retries > 0) {
        	statusCodeAndContent = httpGetRequest(jenkinsBaseUrl.toString() + "queue/api/json");
            retries--;
        	if (statusCodeAndContent.statusCode != 200) {
        		Thread.sleep(500);
        	}
        } 

        int numberOfItemsInQueue = statusCodeAndContent.pathCount("items");
        LOG.debug("found " + numberOfItemsInQueue + " jobs waiting in the queue");
        
        return numberOfItemsInQueue;
    }

    public String getSlaveName() {
        return slaveName;
    }

    public Number getQueueIdForBuildWithXlTestUrl() throws Exception {
        // http://jenkins.dev.bol.com/queue/api/json?tree=items[id,actions[parameters[*]]]&pretty=true
        String url = jenkinsBaseUrl.toString() + "queue/api/json?tree=items[id,actions[parameters[*]]]";
        StatusCodeAndContent statusCodeAndContent = httpGetRequest(url);
        if (statusCodeAndContent.statusCode != 200) {
            LOG.warn("Calling " + url + " did return status code " + statusCodeAndContent.statusCode + ": " + statusCodeAndContent.json);
            return null;
        }
        return extractQueueIdFromBuildQueue((Map<String, List<Map<String, Object>>>) statusCodeAndContent.json);
    }


    public boolean cancelQueuedBuild(Number id) throws Exception {
        // POST! http://localhost:8282/queue/cancelItem?id=2
        StatusCodeAndContent statusCodeAndContent = httpPostRequest(jenkinsBaseUrl.toString() + "queue/cancelItem=" + id);
        return statusCodeAndContent.statusCode < 400;
    }

    public NumberSlavePair getNumberForBuildWithXlTestUrl() throws Exception {
        String url = new URL(jenkinsBaseUrl, "job/" + escape(jobName) + "/api/json?tree=builds[number,result,url,builtOn,actions[parameters[*]]]").toString();
        StatusCodeAndContent statusCodeAndContent = httpGetRequest(url);
		if (statusCodeAndContent.statusCode != 200) {
            LOG.warn("Calling " + url + " did return status code " + statusCodeAndContent.statusCode + ": " + statusCodeAndContent.json);
            return null;
        }
        Number buildNumber = extractBuildNumberFromJob((Map<String, List<Map<String, Object>>>) statusCodeAndContent.json);
        String slave = extractSlaveFromJob((Map<String, List<Map<String, Object>>>) statusCodeAndContent.json);
        return new NumberSlavePair(buildNumber, slave);
    }

    private String escape(String s) throws UnsupportedEncodingException {
        return UrlEscapers.urlFragmentEscaper().escape(s);
    }

    public NumberSlavePair waitForBuildNumber() throws Exception {
    	
    	int retries = 360;
    	while (retries > 0) {
    		retries --;
    		NumberSlavePair numberSlavePair = getNumberForBuildWithXlTestUrl();
	    	if (numberSlavePair == null || numberSlavePair.getNumber() == null) {
	    		// it is not building, it might be in the queue
	    		Number queueIdForBuildWithXlTestUrl = getQueueIdForBuildWithXlTestUrl();
	    		if (queueIdForBuildWithXlTestUrl != null) {
	    			// job is queued. Just wait a bit
	    			LOG.debug("Finding Jenkins Job for id " + uniqueId + " failed. Will retry in 5 seconds as we found it in the Queue with id: " + queueIdForBuildWithXlTestUrl);
	    			Thread.sleep(5000);
	    		} else {
	    			// job seems not building and it is also not queued. We blame jenkins and wait a bit
	    			LOG.debug("Finding Jenkins Job for id " + uniqueId + " failed. It is also not in the Queue. We blame Jenkins. Will retry in 5 seconds anyway");
	    			Thread.sleep(5000);
	    		}
	    		// Note we waited 5000 in any case
	    	} else {
	    		LOG.debug("Finding Jenkins Job for id " + uniqueId + " succeeded. It is: " + numberSlavePair.getNumber());
	    		return numberSlavePair;
	    	}
    	}
    	LOG.error("Finding Jenkins Job for URL " + uniqueId + " failed dramatically. We retried 360 times and now give up");
    	return null;
    }

    public String buildResult() throws Exception {
        String url = new URL(jenkinsBaseUrl, "job/" + escape(jobName) + "/" + buildNumber + "/api/json?tree=number,result").toString();
        StatusCodeAndContent statusCodeAndContent = httpGetRequest(url);

        if (statusCodeAndContent.statusCode != 200) {
            LOG.warn("Calling " + url + " did return status code " + statusCodeAndContent.statusCode + ": " + statusCodeAndContent.json);
            return null;
        } else {
            return (String) BUILD_XLTEST_RESULT.read(statusCodeAndContent.json);
        }
    }

    public int waitForJobToFinish(int timeout) throws Exception {

        LOG.info("Job {} for run {} is waiting in the build queue", jobName, uniqueId);

        try {
            // Check if build is in build queue, wait until it's
            NumberSlavePair buildNumberSlavePair = waitForBuildNumber();
            if (buildNumberSlavePair == null) {
                // queue inspector gave up
                LOG.error("Jenkins Queue Inspector gave up. No clue what to do here");
            }
            buildNumber = buildNumberSlavePair.getNumber();
            slaveName = buildNumberSlavePair.getSlave();

            // We add 50% extra time here, since we expect Jenkins to handle the time out initially.
            final long timeoutTime = System.currentTimeMillis() + ((timeout > 0 ? timeout * MINUTE : ONE_HOUR));

            if (LOG.isInfoEnabled()) {
                LOG.info("Job {} for run {} with time out {} should finish before {}", jobName, uniqueId, timeout, new Date(timeoutTime));
            }

            String buildResult;
            while ((buildResult = buildResult()) == null && !timedOut(timeoutTime)) {
                sleepOnIt();
            }

            if (timedOut(timeoutTime)) {
                LOG.info("Job {} for run {} with time out {} just timed out", jobName, uniqueId, timeout);
                stopBuild();
                return BUILD_TIMEOUT;
            } else {
                if (!SUCCESS.equals(buildResult)) {
//                    sendEvent(makeJobEvent(subRunId, "finished", buildResult == null ? "NULL" : buildResult.toLowerCase(), testSpecification, startMoment, slave, buildNumber));
                    LOG.info("Job {} for run {} did not finish successfully", jobName, uniqueId);
                    return BUILD_FAILURE;
                } else {
//                    sendEvent(makeJobEvent(subRunId, "finished", "success", testSpecification, startMoment, slave, buildNumber));
                    LOG.info("Job {} for run {} did finish successfully", jobName, uniqueId);
                    return BUILD_SUCCESS;
                }
            }
        } catch (InterruptedException ie) {
            cancelJobOnJenkins();
            throw ie;
        }
    }

    private boolean timedOut(long timeoutTime) {
        return System.currentTimeMillis() > timeoutTime;
    }

    private void sleepOnIt() throws InterruptedException {
        // Should release thread and come back at some later time to check the status again.
        Thread.sleep(5000);
    }

    public void stopBuild() throws Exception {
        URL url = new URL(jenkinsBaseUrl, "job/" + escape(jobName) + "/" + buildNumber + "/stop");
        HttpURLConnection conn = (HttpURLConnection) url.openConnection();
        try {
            if (conn.getResponseCode() != 200) {
                LOG.warn("Calling " + url + " did return status code " + conn.getResponseCode() + ": " + CharStreams.toString(new InputStreamReader(conn.getInputStream())));
            }
        } finally {
            conn.disconnect();
        }
    }

    // This function does a best effort attempt to kill jobs running on Jenkins.
    protected void cancelJobOnJenkins() {
        Number queueId = null;
        try {
            queueId = getQueueIdForBuildWithXlTestUrl();
            if (queueId != null && cancelQueuedBuild(queueId)) {
                return;
            }
        } catch (Exception e) {
            LOG.error("Unable to remove build {} from queue", queueId, e);
        }

        try {
            if (this.buildNumber == null) {
                this.buildNumber = waitForBuildNumber().getNumber();
            }
            if (this.buildNumber != null) {
                stopBuild();
            }
        } catch (Exception e) {
            LOG.error("Unable to stop build {}/{}", jobName, this.buildNumber, e);
        }
    }

    Number extractQueueIdFromBuildQueue(Map<String, List<Map<String, Object>>> json) {
        return (Number) extractIdFromJenkinsResponse(json, "items", "id");
    }

    Number extractBuildNumberFromJob(Map<String, List<Map<String, Object>>> json) {
        return (Number) extractIdFromJenkinsResponse(json, "builds", "number");
    }
    
    String extractSlaveFromJob(Map<String, List<Map<String, Object>>> json) {
        return (String) extractIdFromJenkinsResponse(json, "builds", "builtOn");
    }

    private Object extractIdFromJenkinsResponse(Map<String, List<Map<String, Object>>> builds, String groupName, String idName) {
        // { "items": [{ "actions": [{ "parameters": [{ "name": "XLTEST_URL", "value": "http://xxxxx" }, ...]}, ...], "id": 123 }, ... ]}
        // { "builds": [{ "actions": [{ "parameters": [{ "name": "XLTEST_URL", "value": "http://xxxxx" }, ...]}, ...], "number": 123 }, ... ]}
        for (Map<String, Object> build : builds.get(groupName)) {
            List<Map<String, List<Map<String, String>>>> actions = ((List<Map<String, List<Map<String, String>>>>) build.get("actions"));
            if (actions == null || actions.size() == 0) {
                continue;
            }
            List<Map<String, String>> parameters = actions.get(0).get("parameters");
            if (parameters == null || parameters.size() == 0) {
                continue;
            }
            for (Map<String, String> parameter : parameters) {
                if (uniqueIdParameterName.equals(parameter.get("name")) && uniqueId.equals(parameter.get("value"))) {
                    return build.get(idName);
                }
            }
        }
        return null;
    }

    private StatusCodeAndContent httpGetRequest(String url) throws IOException {
        return httpRequest("GET", url);
    }

    private StatusCodeAndContent httpPostRequest(String url) throws IOException {
        return httpRequest("POST", url);
    }

    private StatusCodeAndContent httpRequest(String method, String url) throws IOException {
        HttpURLConnection connection = (HttpURLConnection) new URL(url).openConnection();
        try {
            connection.setDoOutput(true);
            connection.setDoInput(true);
            connection.setInstanceFollowRedirects(false);
            connection.setUseCaches(false);
            connection.setRequestMethod(method);

            if (!Strings.isNullOrEmpty(username)) {
                String userPassword = username + ":" + (password != null ? password : "");
                String encoding = new String(Base64.encodeBytes(userPassword.getBytes()));
                connection.setRequestProperty("Authorization", "Basic " + encoding);
            }

            int statusCode = connection.getResponseCode();


            String content = CharStreams.toString(new InputStreamReader(connection.getInputStream(), Charsets.UTF_8));
            return new StatusCodeAndContent(statusCode, content);
        } finally {
            connection.disconnect();
        }
    }

    private static class StatusCodeAndContent {
        private final JsonProvider jsonProvider;

        private final int statusCode;
        private final Object json;

        public StatusCodeAndContent(int statusCode, String content) {
            this.jsonProvider = JsonProviderFactory.createProvider();
            this.statusCode = statusCode;
            this.json = jsonProvider.parse(content);
        }


        public <T> T path(String path) {
            return JsonPath.read(json, path);
        }

        public int pathCount(String path) {
            Object results = JsonPath.read(json, path);
            if (results instanceof JSONArray) {
                return ((JSONArray) results).size();
            } else if (results instanceof JSONObject) {
                return ((JSONObject) results).size();
            }
            return results != null ? 1 : 0;

        }
    }
}
