package com.xebialabs.xltest.jenkins;

import static com.xebialabs.overthere.ConnectionOptions.ADDRESS;
import static com.xebialabs.overthere.ConnectionOptions.PASSWORD;
import static com.xebialabs.overthere.ConnectionOptions.PORT;
import static com.xebialabs.overthere.ConnectionOptions.USERNAME;
import static java.lang.String.format;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.jboss.resteasy.util.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Strings;
import com.google.common.io.ByteStreams;
import com.google.common.io.Files;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.ConnectionOptions;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.OverthereProcess;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.spi.AddressPortMapper;
import com.xebialabs.overthere.spi.BaseOverthereConnection;
import com.xebialabs.overthere.spi.OverthereConnectionBuilder;
import com.xebialabs.overthere.spi.Protocol;

@Protocol(name = "jenkins")
public class JenkinsConnection extends BaseOverthereConnection implements OverthereConnectionBuilder {
	
	private static final Logger LOG = LoggerFactory.getLogger(JenkinsConnection.class);

    public static final String JENKINS_PROTOCOL = "jenkins";

    /**
     * The name of the job.
     */
    public static final String JOB_NAME = "jobName";

    /**
     * This parameter is set so we can track the job, based on the parameter provided.
     */
    public static final String UNIQUE_ID_PARAMETER_NAME = "uniqueIdParameterName";
    /**
     * Additional parameters needed for teh Job, such as browser, environment, tags, ....
     */
    public static final String JOB_PARAMETERS = "jobParameters";

    // Address always ends with a "/"
    private final String address;

    private final String username;

    private final String password;

    private final String jobName;
    
    private final Map<String, String> jobParameters;

    private final String uniqueIdParameterName;

    private int timeout = JenkinsQueueInspector.ONE_HOUR;
    private File localWorkspaceDirectory;

    public JenkinsConnection(String protocol, ConnectionOptions options, AddressPortMapper mapper) {
        super(protocol, options, mapper, true);
        String addr = options.get(ADDRESS);
        Integer port = options.get(PORT);
        String portAsString = (port == null) ? "" : ":" + port.toString();
        if (addr.endsWith("/")) {
            address = addr.substring(0, addr.length() - 1) + portAsString + "/";
        } else {
            address = addr + portAsString + "/";
        }
        username = options.getOptional(USERNAME);
        password = options.getOptional(PASSWORD);
        jobName = options.get(JOB_NAME);
        jobParameters = options.get(JOB_PARAMETERS);
        uniqueIdParameterName = options.get(UNIQUE_ID_PARAMETER_NAME, UNIQUE_ID_PARAMETER_NAME);
    }


    @Override
    public OverthereFile getFile(String hostPath) {
        // TODO: get path from Jenkins workspace. Watch out for sync from slave to master.
        LOG.info("Looking for path: {}", hostPath);
    	return new JenkinsFile(this, hostPath);
    }

    @Override
    public OverthereFile getFile(OverthereFile parent, String child) {
        LOG.info("Looking for path: {}/{}", parent, child);
        return new JenkinsFile(this, (JenkinsFile) parent, child);
    }

    /**
     * Starts a command with its argument and returns control to the caller.
     *
     * @param commandLine the command line to execute.
     * @return an object representing the executing command or <tt>null</tt> if this is not supported by the host
     *         connection.
     */
    @Override
    public OverthereProcess startProcess(CmdLine commandLine) {
        final String uniqueId = UUID.randomUUID().toString();
        Map<String, Object> parameters = new HashMap<>();
        parameters.put(uniqueIdParameterName, uniqueId);
        if (jobParameters != null) {
        	parameters.putAll(jobParameters);
        }
        int responseCode = buildJob(parameters);

        final JenkinsQueueInspector jenkinsQueueInspector;
        try {
            jenkinsQueueInspector = new JenkinsQueueInspector(new URL("http://" + address), jobName, username, password, uniqueIdParameterName, uniqueId);
        } catch (MalformedURLException e) {
            throw new RuntimeIOException("Unable to inspect Jenkins queue", e);
        }

        return new OverthereProcess() {
            boolean hasExited = false;
            public int exitCode;

            @Override
            public OutputStream getStdin() {
                return new ByteArrayOutputStream();
            }

            @Override
            public InputStream getStdout() {
                return new ByteArrayInputStream(new byte[] {});
            }

            @Override
            public InputStream getStderr() {
                return new ByteArrayInputStream(new byte[] {});
            }

            @Override
            public int waitFor() throws InterruptedException {
                if (!hasExited) {
                    try {
                        exitCode = jenkinsQueueInspector.waitForJobToFinish(timeout);
                    } catch (Exception e) {
                        throw new RuntimeIOException("Error while waiting for job " + jobName + " to finish", e);
                    } finally {
                        hasExited = true;
                    }
                }
                return exitCode;
            }

            @Override
            public void destroy() {
                if (!hasExited) {
                    try {
                        jenkinsQueueInspector.stopBuild();
                    } catch (Exception e) {
                        throw new RuntimeIOException("Unable to force-stop job " + jobName, e);
                    } finally {
                        hasExited = true;
                    }
                }
            }

            @Override
            public int exitValue() throws IllegalThreadStateException {
                if (!hasExited) {
                    throw new IllegalThreadStateException("process hasn't exited");
                }
                return exitCode;

            }
        };
    }


    @Override
    public OverthereConnection connect() {
        return this;
    }

    @Override
    protected void doClose() {
        // no-op
    }

    @Override
    public OverthereFile getFileForTempFile(OverthereFile parent, String name) {
        throw new UnsupportedOperationException("Cannot deal with temp files on " + this);
    }


    @Override
    public String toString() {
        return JENKINS_PROTOCOL + ":" + username + "@" + address + "job/" + jobName;

    }

    protected File downloadJenkinsWorkspace(String hostPath) {
        if (localWorkspaceDirectory != null) {
            return localWorkspaceDirectory;
        }
        HttpURLConnection conn = null;
        try {
            File saveDir = newLocalWorkspaceDirectory(hostPath);
            final String fileName = "xltest-download.zip";
            URL zipUrl = new URL(new URL("http://" + address), format("job/%s/ws/%s/*zip*/%s", jobName, hostPath, fileName));
            conn = (HttpURLConnection) zipUrl.openConnection();
            addHeaders(conn);
            int responseCode = conn.getResponseCode();

            // always check HTTP response code first
            switch (responseCode) {
                case 200:
                    InputStream inputStream = conn.getInputStream();
                    try {
                        extract(saveDir, inputStream);
                    } finally {
                        inputStream.close();
                    }
                    LOG.debug("Done, unzipped in dir: " + localWorkspaceDirectory.getPath());
                    break;
                case 401:
                    if (username != null) {
                        throw new RuntimeIOException(format("Can not log on to Jenkins with user %s", username));
                    } else {
                        throw new RuntimeIOException(format("Authorization is required for Jenkins at %s", address));
                    }
                case 403:
                    throw new RuntimeIOException(format("Not allowed to access workspace folder '%s'", hostPath));
                case 404:
                    throw new RuntimeIOException(format("The folder '%s' could not be found in the workspace for job %s", hostPath, jobName));
                default:
                    throw new RuntimeIOException(format("Unable to download workspace folder '%s'(status code = %d)", hostPath, responseCode));
            }

            return localWorkspaceDirectory;
        } catch (IOException e) {
            localWorkspaceDirectory = null;
            throw new RuntimeIOException(format("Can not connect to Jenkins on URL %s (invalid url?)", address), e);
        } finally {
            if (conn != null) conn.disconnect();
        }
    }

    private File newLocalWorkspaceDirectory(String hostPath) {
        localWorkspaceDirectory = Files.createTempDir();
        File saveDir;
        String parent = new File(hostPath).getParent();
        if (parent != null) {
            saveDir = new File(localWorkspaceDirectory, parent);
        } else {
            saveDir = localWorkspaceDirectory;
            if ("".equals(hostPath) || "/".equals(hostPath) || ".".equals(hostPath)) {
                // Download top-level, Jenkins will package everything in a workspace folder.
                localWorkspaceDirectory = new File(localWorkspaceDirectory, "workspace");
            }
        }
        return saveDir;
    }

    private HttpURLConnection addHeaders(HttpURLConnection conn) {
        conn.setInstanceFollowRedirects(true);
        if (!Strings.isNullOrEmpty(username)) {
            String userPassword = username + ":" + (password != null ? password : "");
            String encoding = new String(Base64.encodeBytes(userPassword.getBytes()));
            conn.setRequestProperty("Authorization", "Basic " + encoding);
        }
        return conn;
    }

    protected void extract(File baseDir, InputStream is) throws IOException {
        final ZipInputStream zis = new ZipInputStream(is);
        ZipEntry entry;
        while ((entry = zis.getNextEntry()) != null) {
            if (!entry.isDirectory()) {
                final File file = new File(baseDir, entry.getName());
                LOG.debug("Copying {} into file {}", entry.getName(), file);
                Files.createParentDirs(file);
                Files.write(ByteStreams.toByteArray(zis), file);
            }
        }
    }

    public int buildJob(Map<String, Object> parameters) {
        HttpURLConnection conn = null;
        try {
            URL buildURL = new URL(new URL("http://" + address), format("job/%s/buildWithParameters?%s", jobName, makeContent(parameters)));
            LOG.info("Kicking off a job on Jenkins using this URL: " + buildURL.toString());
            conn = (HttpURLConnection) buildURL.openConnection();
            conn.setInstanceFollowRedirects(false);
            addHeaders(conn);
            // Need this to trigger the sending of the request
            int responseCode = conn.getResponseCode();
            LOG.info("Kicking off a job on Jenkins resulted in http responseCode: " + responseCode);
            return responseCode;
        } catch (MalformedURLException e) {
            throw new RuntimeIOException("Invalid URL", e);
        } catch (IOException e) {
            throw new RuntimeIOException("Unable to start job", e);
        } finally {
            if (conn != null) conn.disconnect();
        }
    }

    private String makeContent(Map<String, Object> parameters) {
        StringBuilder sb = new StringBuilder();
        if (parameters != null) {
            for (String key : parameters.keySet()) {
                sb.append(key);
                sb.append('=');
                sb.append(parameters.get(key));
                sb.append('&');
            }
        }
        return sb.toString().substring(0, sb.toString().length() - 1);
    }

    public File getAbsoluteFileInLocalWorkspaceDirectory(String path) {
        return new File(downloadJenkinsWorkspace(path), path);
    }
}
