package com.xebialabs.xltest.jenkins;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.tasker.TaskStep;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.domain.TestRunId;
import com.xebialabs.xltest.domain.TestSetDefinition;
import com.xebialabs.xltest.plan.TestPlan;
import com.xebialabs.xltest.repository.EventNotifier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import javax.ws.rs.core.UriBuilder;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import static com.xebialabs.xltest.jenkins.JenkinsQueueInspector.XLTEST_URL;
import static java.lang.String.format;

/**
 * Execute jobs on a Jenkins host.
 *
 * Each test run id is enriched with a unique id, so the "finish" notification can be properly processed.
 */
@SuppressWarnings("serial")
@Metadata(description = "Jenkins Test run base type", root = Metadata.ConfigurationItemRoot.APPLICATIONS, virtual = true)
public class JenkinsTestRun extends TestRun {
    private static final Logger LOG = LoggerFactory.getLogger(JenkinsTestRun.class.getName());
    // We want to prevent all steps from scheduling jobs on Jenkins at the same time, hence we limit the number of
    // steps actively checking for a slot on Jenkins.
    private static final Object SCHEDULE_LOCK = new Object();
    public static final int MINUTE = 60 * 1000;
    public static final int ONE_HOUR = 60 * MINUTE;
    public static final String SUCCESS = "SUCCESS";

    @Property
    private String jenkinsUri;
    @Property
    private String jobName;
    @Property
    private int maxQueueSize;
    @Property(description = "target operating system", hidden = true, defaultValue = "UNIX")
    private OperatingSystemFamily targetOsType;

    @Autowired
    private transient EventNotifier eventNotifier;

    private transient long testChildRunId = 0;

    @Override
    protected List<StepState> getSteps(TestPlan testPlan) {
        return Lists.<StepState>newArrayList(
                new TaskStep(new JenkinsExecutionStep(testPlan))
        );
    }

    private void updateParameterMapWithTestRunProperties(Map<String, Object> parameters) {
        for (PropertyDescriptor pd : this.getType().getDescriptor().getPropertyDescriptors()) {
            Object value = pd.get(this);
            if (value != null) {
                parameters.put(pd.getName(), value);
            }
        }
    }

    private void updateParameterMapWithTestSetDefinitionProperties(TestSetDefinition testSetDefinition, Map<String, Object> parameters) {
        for (PropertyDescriptor pd : testSetDefinition.getType().getDescriptor().getPropertyDescriptors()) {
            Object value = pd.get(testSetDefinition);
            if (value != null) {
                parameters.put(pd.getName(), value);
            }
        }
    }

    private void ensureJobQueueDoesNotExceed(int numberOfItems) throws Exception {
        JenkinsQueueInspector inspector = newQueueInspector();
        while (inspector.inspectNumberOfJobsInBuildQueue() >= numberOfItems) {
            sleepOnIt();
        }
    }

    private void waitForJobToFinish(TestRunId subRunId, TestSetDefinition testSet, String xlTestUrl) throws Exception {
        int timeout = testSet.getTimeout();
        Number buildNumber = null;
        JenkinsQueueInspector queueInspector = newQueueInspector();

        LOG.info("Job {} for run {} is waiting in the build queue", jobName, xlTestUrl);

        try {
            // Check if build is in build queue, wait until it's
        	NumberSlavePair buildNumberSlavePair = queueInspector.waitForNumberForBuildWithXlTestUrl(jobName, xlTestUrl);
            buildNumber = buildNumberSlavePair.getNumber();
            String slave = buildNumberSlavePair.getSlave();

            // We add 50% extra time here, since we expect Jenkins to handle the time out initially.
            final long timeoutTime = System.currentTimeMillis() + ((timeout > 0 ? timeout * MINUTE : ONE_HOUR));

            if (LOG.isInfoEnabled()) {
                LOG.info("Job {} for run {} with time out {} should finish before {}", jobName, xlTestUrl, timeout, new Date(timeoutTime));
            }

            Event startedEvent = makeJobEvent(subRunId, "started", testSet, slave);
            Number startMoment = startedEvent.get(Event.TIMESTAMP);

            sendEvent(startedEvent);

            String buildResult;
            while ((buildResult = queueInspector.buildResult(jobName, buildNumber)) == null && !timedOut(timeoutTime)) {
                sleepOnIt();
            }

            if (timedOut(timeoutTime)) {
                LOG.info("Job {} for run {} with time out {} just timed out", jobName, xlTestUrl, timeout);
                queueInspector.stopBuild(jobName, buildNumber);
                sendEvent(makeJobEvent(subRunId, "finished", "timeout", testSet, startMoment, slave));
            } else {
                if (!SUCCESS.equals(buildResult)) {
                    LOG.info("Job {} for run {} did not finish successfully", jobName, xlTestUrl);
                    sendEvent(makeJobEvent(subRunId, "finished", buildResult == null ? "NULL" : buildResult.toLowerCase(), testSet, startMoment, slave));
                } else {
                    sendEvent(makeJobEvent(subRunId, "finished", "success", testSet, startMoment, slave));
                    LOG.info("Job {} for run {} did finish successfully", jobName, xlTestUrl);
                }
            }
        } catch (InterruptedException ie) {
            cancelJobOnJenkins(xlTestUrl, buildNumber, queueInspector);
            throw ie;
        }
    }

    // This function does a best effort attempt to kill jobs running on Jenkins.
    private void cancelJobOnJenkins(String xlTestUrl, Number buildNumber, JenkinsQueueInspector queueInspector) {
        Number queueId = null;
        try {
            queueId = queueInspector.getQueueIdForBuildWithXlTestUrl(xlTestUrl);
            if (queueId != null && queueInspector.cancelQueuedBuild(queueId)) {
                return;
            }
        } catch (Exception e) {
            LOG.error("Unable to remove build {} from queue", queueId, e);
        }

        try {
            if (buildNumber == null) {
                buildNumber = queueInspector.waitForNumberForBuildWithXlTestUrl(jobName, xlTestUrl).getNumber();
            }
            if (buildNumber != null) {
                queueInspector.stopBuild(jobName, buildNumber);
            }
        } catch (Exception e) {
            LOG.error("Unable to stop build {}/{}", jobName, buildNumber, e);
        }
    }

    private void sendEvent(Event event) {
        eventNotifier.notify(new TestRunId(getName()), event);
    }

    private Event makeJobEvent(TestRunId subRunId, String status, String reason, TestSetDefinition testSet, Number startMoment, String slave) {
    	Map<String, Object> props = new TreeMap();
        props.put("type", "jobStatus");
        props.put("status", status);
        props.put("subRunId", subRunId.toString());
        props.put("run_id", getName());
        if (slave != null) {
        	props.put("slave", slave);
        }
        if (reason != null) {
        	props.put("reason", reason);
        }
        if (startMoment != null) {
        	props.put("started", startMoment);
        }

        // explicitly add all properties of the test run so we can optimize queries
        for (PropertyDescriptor descriptor : this.getType().getDescriptor().getPropertyDescriptors()) {
            Object property = this.getProperty(descriptor.getName());
            if (property instanceof Number || property instanceof Boolean) {
                props.put(descriptor.getName(), property);
            } else if (property != null) {
                props.put(descriptor.getName(), property.toString());
            }
        }

        // explicitly add all properties of the testSetDef as these are NOT in the repo (we're working sliced!)
        for (PropertyDescriptor descriptor : testSet.getType().getDescriptor().getPropertyDescriptors()) {
            Object property = testSet.getProperty(descriptor.getName());
            if (property instanceof Number || property instanceof Boolean) {
                props.put(descriptor.getName(), property);
            } else if (property != null) {
                props.put(descriptor.getName(), property.toString());
            }
        }
        return new Event(props);
    }
    
    private Event makeJobEvent(TestRunId subRunId, String status, TestSetDefinition testSet, String slave) {
    	return makeJobEvent(subRunId, status, null, testSet, null, slave);
    }

	private boolean timedOut(long timeoutTime) {
        return System.currentTimeMillis() > timeoutTime;
    }

    private JenkinsJobScheduler newJobScheduler() {
        try {
            return new JenkinsJobScheduler(new URL(jenkinsUri));
        } catch (MalformedURLException e) {
            throw new RuntimeException("Invalid url: " + jenkinsUri, e);
        }
    }

    private JenkinsQueueInspector newQueueInspector() {
        try {
            return new JenkinsQueueInspector(new URL(jenkinsUri));
        } catch (MalformedURLException e) {
            throw new RuntimeException("Invalid url: " + jenkinsUri, e);
        }
    }

    private void sleepOnIt() {
        // Should release thread and come back at some later time to check the status again.
        try {
            Thread.sleep(2000);
        } catch (InterruptedException e) {
            // NOOP. No need to throw exception when sleep is interrupted.
        }
    }

    private synchronized TestRunId generateChildRunId() {
        return new TestRunId(String.valueOf(testChildRunId++));
    }

    private URI getTestRunUrl(URI absolutePath, TestRunId testRunId) {
        return UriBuilder.fromUri(absolutePath).path(testRunId.toString()).build();
    }

    public class JenkinsExecutionStep implements Step {

        private TestPlan testPlan;

        public JenkinsExecutionStep(TestPlan testPlan) {
            this.testPlan = testPlan;
        }

        @Override
        public int getOrder() {
            return 0;
        }

        @Override
        public String getDescription() {
            return format("Executing Test set '%s' on local environment", getTestSetDefinition().getId());
        }

        @Override
        public StepExitCode execute(ExecutionContext executionContext) throws Exception {
            TestRunId subRunId = generateChildRunId();

            LOG.info("Scheduling Jenkins job for " + getTestRunId());

            Map<String, Object> parameters = new TreeMap();
            URI xlTestUrl = getTestRunUrl(getUri(), subRunId);
            updateParameterMapWithTestRunProperties(parameters);
            updateParameterMapWithTestSetDefinitionProperties(testPlan.getTestSet(), parameters);
            parameters.put(XLTEST_URL, xlTestUrl);
            parameters.put("commandLine", testPlan.getCommandLine().toCommandLine(targetOsType, false));

            // In order to limit the queue size, we need to check and schedule in one "atomic" action.
            synchronized (SCHEDULE_LOCK) {
                JenkinsJobScheduler scheduler = newJobScheduler();

                ensureJobQueueDoesNotExceed(maxQueueSize);
                scheduler.scheduleJenkinsJobWithParameters(jobName, parameters);
            }

            // TODO: -AJM- We need a dingetje that keeps track of the state of our Jenkins runs.
            waitForJobToFinish(subRunId, testPlan.getTestSet(), xlTestUrl.toString());

            return StepExitCode.SUCCESS;

        }

    }

}
