package com.xebialabs.xltest.jenkins;

import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.spi.JsonProviderFactory;
import com.xebialabs.xltest.utils.JsonOverHttpHelper;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URL;
import java.util.List;
import java.util.Map;

/**
 * Sends events to XL Test with information found in the wiki pages.
 */
public class JenkinsQueueInspector {
    private static final Logger LOG = LoggerFactory.getLogger(JenkinsQueueInspector.class.getName());

    public static final String XLTEST_URL = "XLTEST_URL";

    static final JsonPath BUILD_XLTEST_RESULT = JsonPath.compile("$.result");

    // the jenkins base url is something like: "http://localhost:8080"
    private URL jenkinsBaseUrl;

    public JenkinsQueueInspector(URL url) {
        this.jenkinsBaseUrl = url;
    }

    public int inspectNumberOfJobsInBuildQueue() throws Exception {
        JsonOverHttpHelper httpHelper = new JsonOverHttpHelper();
        httpHelper.httpGetRequest(jenkinsBaseUrl.toString() + "/queue/api/json");

        int numberOfItemsInQueue = httpHelper.pathCount("items");
        LOG.info("found " + numberOfItemsInQueue + " jobs waiting in the queue");
        return numberOfItemsInQueue;
    }

    public Number getQueueIdForBuildWithXlTestUrl(String xlTestUrl) throws Exception {
        JsonOverHttpHelper httpHelper = new JsonOverHttpHelper();
        // http://jenkins.dev.bol.com/queue/api/json?tree=items[id,actions[parameters[*]]]&pretty=true
        String url = jenkinsBaseUrl.toString() + "/queue/api/json?tree=items[id,actions[parameters[*]]]";
        httpHelper.httpGetRequest(url);
        if (httpHelper.statusCode() != 200) {
            LOG.warn("Calling " + url + " did return status code " + httpHelper.statusCode() + ": " + httpHelper.content());
            return null;
        }
        return extractQueueIdFromBuildQueue(httpHelper.content(), xlTestUrl);
    }


    public boolean cancelQueuedBuild(Number id) throws Exception {
        // POST! http://localhost:8282/queue/cancelItem?id=2
        JsonOverHttpHelper httpHelper = new JsonOverHttpHelper();
        httpHelper.httpPostRequest(jenkinsBaseUrl.toString() + "/queue/cancelItem=" + id, "");
        return httpHelper.statusCode() < 400;
    }

    public NumberSlavePair getNumberForBuildWithXlTestUrl(String jobName, String xlTestUrl) throws Exception {
        JsonOverHttpHelper httpHelper = new JsonOverHttpHelper();
        String url = jenkinsBaseUrl.toString() + "/job/" + jobName + "/api/json?tree=builds[number,result,url,builtOn,actions[parameters[*]]]";
        httpHelper.httpGetRequest(url);
        String content = httpHelper.content();
		if (httpHelper.statusCode() != 200) {
            LOG.warn("Calling " + url + " did return status code " + httpHelper.statusCode() + ": " + content);
            return null;
        }
        Number buildNumber = extractBuildNumberFromJob(content, xlTestUrl);
        String slave = extractSlaveFromJob(content, xlTestUrl);
        return new NumberSlavePair(buildNumber, slave);
    }

    public NumberSlavePair waitForNumberForBuildWithXlTestUrl(String jobName, String xlTestUrl) throws Exception {
    	
    	Number number = null;
    	int retries = 360;
    	while (retries > 0) {
    		retries --;
    		NumberSlavePair numberSlavePair = getNumberForBuildWithXlTestUrl(jobName, xlTestUrl);
	    	number = numberSlavePair.getNumber();
	    	if (number == null) {
	    		// it is not building, it might be in the queue
	    		Number queueIdForBuildWithXlTestUrl = getQueueIdForBuildWithXlTestUrl(xlTestUrl);
	    		if (queueIdForBuildWithXlTestUrl != null) {
	    			// job is queued. Just wait a bit
	    			LOG.error("Finding Jenkins Job for URL " + xlTestUrl + " failed. Will retry in 5 seconds as we found it in the Queue with id: " + queueIdForBuildWithXlTestUrl);
	    			Thread.sleep(5000);
	    		} else {
	    			// job seems not building and it is also not queued. We blame jenkins and wait a bit
	    			LOG.error("Finding Jenkins Job for URL " + xlTestUrl + " failed. It is also not in the Queue. We blame Jenkins. Will retry in 5 seconds anyway");
	    			Thread.sleep(5000);
	    		}
	    		// Note we waited 5000 in any case
	    	} else {
	    		LOG.error("Finding Jenkins Job for URL " + xlTestUrl + " succeeded. It is: " + number);
	    		return numberSlavePair;
	    	}
    	}
    	LOG.error("Finding Jenkins Job for URL " + xlTestUrl + " failed dramatically. We retried 360 times and now give up");
    	return null;
    	
    	
//        // sometimes we get inappropriate results back from jenkins resulting in a null
//        Number number = null;
//        int retries = 100;
//        boolean haveSeenBuildInTheQueue = false;
//        while (number == null && retries > 0) {
//            retries--;
//            try {
//                number = getNumberForBuildWithXlTestUrl(jobName, xlTestUrl);
//            } catch (Throwable t) {
//                LOG.error("Could not retrieve build number from Jenkins build queue. Will retry.", t);
//            }
//            if (number == null) {
//                Thread.sleep(2000); // wilco tijdens deze sleep kan ie de queue uit en gebuild worden
//                // As long as the job is still in the queue, it makes no sense checking for a build number
//                if (!haveSeenBuildInTheQueue) {
//                    try {
//                        while (getQueueIdForBuildWithXlTestUrl(xlTestUrl) != null) { // wilco hangt als conditie boven waar is
//                            haveSeenBuildInTheQueue = true;
//                            Thread.sleep(2000); // wilco in deze tijd kan ie in de queue komen en er al weer uit gaan
//                        }
//                    } catch (Throwable t) {
//                        LOG.error("Could not retrieve Jenkins' build queue. Will retry.", t);
//                    }
//                }
//            }
//        }
//        if (number == null) {
//            throw new RuntimeException("Unable to find appropriate Jenkins job url for job: " + jobName + " and XL Test url: " + xlTestUrl + " so can't stop the job. I retried 100 times");
//        }
//        return number;
    }

    public String buildResult(String jobName, Number buildNumber) throws Exception {
        JsonOverHttpHelper httpHelper = new JsonOverHttpHelper();

        String url = jenkinsBaseUrl.toString() + "/job/" + jobName + "/" + buildNumber + "/api/json?tree=number,result";
        httpHelper.httpGetRequest(url);

        if (httpHelper.statusCode() != 200) {
            LOG.warn("Calling " + url + " did return status code " + httpHelper.statusCode() + ": " + httpHelper.content());
            return null;
        } else {
            return (String) BUILD_XLTEST_RESULT.read(httpHelper.content());
        }
    }

    public void stopBuild(String jobName, Number buildNumber) throws Exception {
        JsonOverHttpHelper httpHelper = new JsonOverHttpHelper();
        String url = jenkinsBaseUrl.toString() + "/job/" + jobName + "/" + buildNumber + "/stop";
        httpHelper.httpGetRequest(url);
        if (httpHelper.statusCode() != 200) {
            LOG.warn("Calling " + url + " did return status code " + httpHelper.statusCode() + ": " + httpHelper.content());
        }
    }

    Number extractQueueIdFromBuildQueue(String json, String xlTestUrl) {
        return (Number) extractIdFromJenkinsResponse(json, xlTestUrl, "items", "id");
    }


    Number extractBuildNumberFromJob(String json, String xlTestUrl) {
        return (Number) extractIdFromJenkinsResponse(json, xlTestUrl, "builds", "number");
    }
    
    String extractSlaveFromJob(String json, String xlTestUrl) {
        return (String) extractIdFromJenkinsResponse(json, xlTestUrl, "builds", "builtOn");
    }

    private Object extractIdFromJenkinsResponse(String json, String xlTestUrl, String groupName, String idName) {
        // { "items": [{ "actions": [{ "parameters": [{ "name": "XLTEST_URL", "value": "http://xxxxx" }, ...]}, ...], "id": 123 }, ... ]}
        // { "builds": [{ "actions": [{ "parameters": [{ "name": "XLTEST_URL", "value": "http://xxxxx" }, ...]}, ...], "number": 123 }, ... ]}
        Map<String, List<Map<String, Object>>> builds = (Map<String, List<Map<String, Object>>>) JsonProviderFactory.createProvider().parse(json);
        for (Map<String, Object> build : builds.get(groupName)) {
            List<Map<String, List<Map<String, String>>>> actions = ((List<Map<String, List<Map<String, String>>>>) build.get("actions"));
            if (actions == null || actions.size() == 0) {
                continue;
            }
            List<Map<String, String>> parameters = actions.get(0).get("parameters");
            if (parameters == null || parameters.size() == 0) {
                continue;
            }
            for (Map<String, String> parameter : parameters) {
                if (JenkinsQueueInspector.XLTEST_URL.equals(parameter.get("name")) && xlTestUrl.equals(parameter.get("value"))) {
                    return build.get(idName);
                }
            }
        }
        return null;
    }
}
