package com.xebialabs.xltest.fitnesse;

import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.EventHandler;
import com.xebialabs.xltest.domain.Importable;
import fitnesse.FitNesseContext;
import fitnesse.reporting.history.*;
import fitnesse.wiki.PageData;
import fitnesse.wiki.PathParser;
import fitnesse.wiki.WikiPage;
import fitnesse.wiki.fs.FileSystemPage;
import fitnesse.wiki.fs.FileSystemPageFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import util.FileUtil;

import java.io.File;
import java.util.*;

import static java.lang.String.format;

public class PageHistoryExtractor {
    private static final Logger LOG = LoggerFactory.getLogger(PageHistoryExtractor.class.getName());

    private final File resultsDirectory;
    private final FileSystemPage root;

    public PageHistoryExtractor() {
        this(new File("./FitNesseRoot"));
    }
    public PageHistoryExtractor(File fitNesseRoot) {
        resultsDirectory = new File(fitNesseRoot, "/files/" + FitNesseContext.testResultsDirectoryName);
        root = new FileSystemPageFactory().makePage(fitNesseRoot, "FitNesseRoot", null);
    }


    public List<Importable> tellMeAboutAllSuites() throws Exception {
        List<Importable> toImport = new ArrayList<>();


        File[] resultFiles = resultsDirectory.listFiles();
        if (resultFiles == null) {
            return Collections.emptyList();
        }

        for (File pageNamedFile : resultFiles) {
            if (!pageNamedFile.isDirectory()) {
                continue;
            }

            // check if it contains test results or suite results
            String pageName = pageNamedFile.getName();
            LOG.info("Will try to make report for " + pageName);

            PageHistory pageHistory = getPageHistory(resultsDirectory, pageName);

            if (pageHistoryIsSuite(pageHistory)) {
                for (Date date : pageHistory.datesInChronologicalOrder()) {
                    ExecutionReport executionReport = getExecutionReport(pageHistory, date);
                    if (executionReport instanceof SuiteExecutionReport) {
                        toImport.add(new FitNesseSuite((SuiteExecutionReport) executionReport, pageName));
                    }
                }
            }
        }

        LOG.info("All import options found");
        return toImport;
    }

    public void tellMeAboutReport(UUID testRunId, SuiteExecutionReport suiteExecutionReport, EventHandler eventHandler) throws Exception {
        for (SuiteExecutionReport.PageHistoryReference reference : suiteExecutionReport.getPageHistoryReferences()) {
        	// get tags
        	WikiPage thePage = root.getPageCrawler().getPage(PathParser.parse(reference.getPageName()));
        	Set<String> tags = makeTags(thePage);
        	String firstErrorMessage = findFirstErrorMessage(resultsDirectory, reference.getPageName(), new Date(reference.getTime()));

            Event event = new Event(testRunId, Event.FUNCTIONAL_RESULT, new TestSummary(
                    Event.normalizeName(reference.getPageName(), '.'),
                    tags,
                    reference.getTime(),
                    reference.getRunTimeInMillis(),
                    reference.getTestSummary().getRight(),
                    reference.getTestSummary().getWrong(),
                    reference.getTestSummary().getWrong() + reference.getTestSummary().getExceptions() > 0 ? "FAILED" : "PASSED",
                    reference.getTestSummary().getExceptions(),
                    firstErrorMessage).toMap());
            eventHandler.onReceive(event);
        }
    }

    private Set<String> makeTags(WikiPage page) {
        if (page == null || page.isRoot()) {
            return set();
        }

        String tags = page.getData().getAttribute(PageData.PropertySUITES);
        Set<String> tagSet = isNotBlank(tags) ? set(tags.split(",")) : set();
        tagSet.addAll(makeTags(page.getParent()));
        return tagSet;
    }

    private String findFirstErrorMessage(File resultsDirectory, String pageName, Date date) throws Exception {
        TestExecutionReport testExecutionReport = getExecutionReport(getPageHistory(resultsDirectory, pageName), date);

        return findFirstErrorMessage(testExecutionReport);
    }

    private String findFirstErrorMessage(TestExecutionReport testExecutionReport) {
        TestExecutionReport.TestResult testResult = testExecutionReport.getResults().get(0);

        for (TestExecutionReport.InstructionResult instructionResult : testResult.getInstructions()) {
            for (TestExecutionReport.Expectation expectation : instructionResult.getExpectations()) {
                if ("fail".equals(expectation.getStatus())) {
                    return format("Actual: '%s'; Expected: '%s'", expectation.getActual(), expectation.getExpected());
                } if ("error".equals(expectation.getStatus())) {
                    return expectation.getEvaluationMessage();
                }
            }
        }
        return null;
    }

    private <T extends ExecutionReport> T getExecutionReport(PageHistory pageHistory, Date date) throws Exception {
        TestResultRecord testResultRecord = pageHistory.get(date);
        return (T) ExecutionReport.makeReport(FileUtil.getFileContent(testResultRecord.getFile()));
    }

    private PageHistory getPageHistory(File resultsDirectory, String pageName) {
        TestHistory history = new TestHistory();
        history.readPageHistoryDirectory(resultsDirectory, pageName);
        return history.getPageHistory(pageName);
    }

    private boolean pageHistoryIsSuite(PageHistory pageHistory) throws Exception {
        ExecutionReport report = getExecutionReport(pageHistory, pageHistory.getLatestDate());
        return report instanceof SuiteExecutionReport;
    }

    public static Set<String> set(String... tags) {
        Set<String> tagSet = new TreeSet();
        for (String tag : tags) {
            String trimmed = tag.trim();
            if (!"".equals(trimmed)) {
                tagSet.add(trimmed);
            }
        }
        return tagSet;
    }

    private boolean isNotBlank(String tags) {
        return tags != null && !"".equals(tags);
    }

    private class FitNesseSuite implements Importable {
        private final SuiteExecutionReport executionReport;
        private final String pageName;

        public FitNesseSuite(SuiteExecutionReport executionReport, String pageName) {
            this.executionReport = executionReport;
            this.pageName = pageName;
        }

        @Override
        public void doImport(UUID testRunId, EventHandler eventHandler) {
            LOG.info("Importing {}", executionReport);
            long startTime = executionReport.getDate().getTime();
            try {
                eventHandler.onReceive(new Event(testRunId, Event.IMPORT_STARTED, Event.props(
                        "fileName", executionReport.getRootPath(),
                        "lastModified", executionReport.getDate().getTime(),
                        "name", pageName
                )));

                tellMeAboutReport(testRunId, executionReport, eventHandler);
            } catch (Exception e) {
                LOG.error("Unable to import test run " + pageName + " for " + executionReport, e);
            } finally {
                long duration = executionReport.getTotalRunTimeInMillis();
                try {
                    eventHandler.onReceive(new Event(testRunId, Event.IMPORT_FINISHED, Event.props(
                            "duration", duration
                    )));
                } catch (Exception e) {
                    LOG.error("Unable to handle finish event properly. Such a shame!", e);
                }

            }
        }
    }
}
