package com.xebialabs.xltest.fitnesse;

import fitnesse.http.MockRequest;
import fitnesse.http.Request;
import fitnesse.responders.run.SuiteResponder;
import fitnesse.testrunner.SuiteContentsFinder;
import fitnesse.testrunner.SuiteFilter;
import fitnesse.wiki.*;
import fitnesse.wiki.fs.FileSystemPageFactory;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.*;
import java.util.logging.Logger;

public class WikiPageExtractor {
    private static final Logger LOG = Logger.getLogger(WikiPageExtractor.class.getName());

    // This is the special field that matches a table row with a test case.
    public static final String PAGE_NAME_TABLE_COLUMN_NAME = "id_link";

    private final Feedback feedback;
    private String workingDir = ".";
    private String fitNesseRootDir = "FitNesseRoot";

    public WikiPageExtractor(Feedback feedback) {
        this.feedback = feedback;
    }

    public WikiPageExtractor withWorkingDir(String workingDir) {
        this.workingDir = workingDir;
        return this;
    }

    public WikiPageExtractor withFitNesseRootDir(String fitNesseRootDir) {
        this.fitNesseRootDir = fitNesseRootDir;
        return this;
    }

    public void tellMeAboutSuite(String suitePath, String filterOptions, String eventType) {
        Request request = makeRequest(suitePath, filterOptions);
        WikiPage root = new FileSystemPageFactory().makeRootPage(workingDir, fitNesseRootDir);
        for (WikiPage page: findWikiPages(request, root)) {
            String pageName = page.getPageCrawler().getFullPath().toString();
            feedback.found(new PageSummary(pageName, makeTags(page), makeExtraProperties(pageName, page), eventType));
        }
    }

    private LinkedList<WikiPage> findWikiPages(Request request, WikiPage root) {
        String suitePath = request.getResource();
        SuiteFilter filter = SuiteResponder.createSuiteFilter(request, suitePath);
        WikiPage suitePage = root.getPageCrawler().getPage(PathParser.parse(suitePath));
        SuiteContentsFinder suiteTestFinder = new SuiteContentsFinder(suitePage, filter, root);
        return suiteTestFinder.getAllPagesToRunForThisSuite();
    }

    private Request makeRequest(String suitePath, String filterOptions) {
        Request request = new MockRequest();
        if (filterOptions != null) {
            request.parseRequestUri("/" + suitePath + "?" + filterOptions);
        } else {
            request.parseRequestUri("/" + suitePath);
        }
        return request;
    }

    private Map<String, String> makeExtraProperties(String pageName, WikiPage page) {
        if (page.isRoot()) {
            return Collections.emptyMap();
        }

        TestCaseTable table = TestCaseTable.fromPage(page);
        for (Map<String, String> row : table.asKeyValuePairs()) {
            if (row.containsKey(PAGE_NAME_TABLE_COLUMN_NAME) && pageName.equals(row.get(PAGE_NAME_TABLE_COLUMN_NAME))) {
                return row;
            }
        }
        return makeExtraProperties(pageName, page.getParent());
    }

    private Set<String> makeTags(WikiPage page) {
        if (page.isRoot()) {
            return set();
        }

        String tags = page.getData().getAttribute(PageData.PropertySUITES);
        Set<String> tagSet = isNotBlank(tags) ? set(tags.split(",")) : set();
        tagSet.addAll(makeTags(page.getParent()));
        return tagSet;
    }

    public static Set<String> set(String... tags) {
        Set<String> tagSet = new TreeSet<String>();
        for (String tag : tags) {
            String trimmed = tag.trim();
            if (!"".equals(trimmed)) {
                tagSet.add(trimmed);
            }
        }
        return tagSet;
    }

    private boolean isNotBlank(String tags) {
        return tags != null && !"".equals(tags);
    }

    /**
     * Query FitNesse wiki pages. "test case" events are sent to the provided callback url.
     *
     * @param args suite name and callback url.
     */
    public static void main(String... args) {
        String suiteName = args[0];
        String eventType = args[1];
        String filterOptions = args.length > 2 ? args[2] : null;
        String url = System.getenv("XLTEST_URL");

        LOG.info("Starting Wiki page extractor...");
        LOG.info("Suite: " + suiteName);
        LOG.info("Sending " + eventType + " events to " + url);

        try {
        	// use FeedbackEventPrinter for testing instead of FeedbackEventSender
            WikiPageExtractor extractor = new WikiPageExtractor(new FeedbackEventPrinter(new URL(url)));
            extractor.tellMeAboutSuite(suiteName, filterOptions, eventType);
        } catch (MalformedURLException e) {
            LOG.severe("Invalid URL: " + url);
        }
    }
}
