package com.xebialabs.xltest.cucumber;

import com.jayway.jsonpath.InvalidPathException;
import com.jayway.jsonpath.spi.JsonProviderFactory;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.EventHandler;
import com.xebialabs.xltest.domain.Importable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.util.*;

import static com.jayway.jsonpath.JsonPath.read;

public class CucumberReportJsonImporter implements Importable {

	private static final Logger LOG = LoggerFactory.getLogger(CucumberReportJsonImporter.class.getName());


    private final OverthereFile cucumberReportJsonFile;

    public CucumberReportJsonImporter(OverthereFile cucumberReportJsonFile) {
        this.cucumberReportJsonFile = cucumberReportJsonFile;
    }

    @Override
    public void doImport(UUID testRunId, EventHandler eventHandler) throws Exception {
    	long started = cucumberReportJsonFile.lastModified();
        eventHandler.onReceive(new Event(testRunId, Event.IMPORT_STARTED, Event.props(
                "lastModified", started,
                "fileName", cucumberReportJsonFile.getPath()
        )));

        List<Object> features;
        try (InputStream fis = cucumberReportJsonFile.getInputStream()) {
            features = (List<Object>) JsonProviderFactory.createProvider().parse(fis);
        }

        long totalDuration = 0;
        for (Object feature : features) {
        	long duration = tellAboutFeature((Map<String, Object>) feature, testRunId, eventHandler);
        	totalDuration = totalDuration + duration;
        }

        eventHandler.onReceive(new Event(testRunId, Event.IMPORT_FINISHED, Event.props(
                "duration", totalDuration
        )));

        LOG.info("Generated start / finish event for file: " + cucumberReportJsonFile.getPath());
    }

    private long tellAboutFeature(Map<String, Object> feature, UUID testRunId, EventHandler eventHandler) throws Exception {
        String featureId = read(feature, "id");
        String featureName = read(feature, "name");
        List<String> featureTags = readTags(feature);

        List<Map<String, Object>> scenarios = read(feature, "elements[?(@.type == 'scenario')]");

        long totalDurationInMillis = 0;
        for (Map<String, Object> scenario : scenarios) {
        	long duration = scenarioDuration(scenario);
            String scenarioResult = scenarioResult(scenario);
            Event event = new Event(testRunId, Event.FUNCTIONAL_RESULT, Event.props(
                    "result", scenarioResult,
                    "duration", duration,
                    "featureId", featureId,
                    "id", read(scenario, "id"),
                    "name", featureName + ";" + read(scenario, "name") + ";" + read(scenario, "line"),
                    "tags", scenarioTags(scenario, featureTags),
                    "steps", read(scenario, "steps")
            ));
            eventHandler.onReceive(event);
            totalDurationInMillis = totalDurationInMillis + duration;
        }
        return totalDurationInMillis;
    }

    private List<String> readTags(Map<String, Object> feature) {
        try {
            return read(feature, "$.tags[*].name");
        } catch (InvalidPathException e) {
            return Collections.emptyList();
        }
    }

    private String scenarioResult(Map<String, Object> scenario) {
        List<String> results = read(scenario, "$.steps[*].result.status");
        for (String result : results) {
            if (!"passed".equals(result)) {
                return "FAILED";
            }
        }
        return "PASSED";
    }

    private long scenarioDuration(Map<String, Object> scenario) {
        List<Number> durations = read(scenario, "$.steps[*].result.duration");
        long total = 0;
        for (Number duration : durations) {
            total = total + duration.longValue();
        }
        return total / 1000;

    }

    private List<String> scenarioTags(Map<String, Object> scenario, List<String> featureTags) {
        List<String> tags = new ArrayList<>(featureTags);
        List<String> scenarioTags = readTags(scenario);
        tags.addAll(scenarioTags);
        return tags;
    }
}
