
TYPE = 0
X_AXIS = 1
RESULT = 2
PAGE_NAME = 3
N_TESTS = 2
SUITE_NAME = 3

def events_to_tuples(events, xaxis):
    tuples = []
    for ev in events:
        if ev.type == 'functionalResult':
            tuples.append((ev.type, ev.getProperties().get(xaxis), ev.get('result'), ev.get('pageName')))
        elif ev.type == 'jobStatus' and ev.get('reason') == 'timeout':
            tuples.append(('timeout', ev.getProperties().get(xaxis) or 'timed out', ev.getProperties().get('ntests') or 0, ev.get('suiteName')))
    return tuples


def overlay_data(events, oldEvents):
    def dict_type_filter(evs, type):
        m = {}
        for e in evs:
            if e[TYPE] == type:
                m[e[PAGE_NAME]] = e
        return m

    resultMap = dict_type_filter(events, 'result')
    results = dict_type_filter(oldEvents, 'result')
    timeoutMap = dict_type_filter(events, 'timeout')
    oldTimeoutMap = dict_type_filter(oldEvents, 'timeout')

    # Update with fresh results
    results.update(resultMap)

    # Remove results for cases with a timeout
    for suiteName, ev in timeoutMap.iteritems():
        for pageName in results.keys():
            if pageName.startswith(suiteName):
                del results[pageName]

    # Add timeouts that have no fresh results associated
    for suiteName, ev in oldTimeoutMap.iteritems():
        for pageName in results.keys():
            if pageName.startswith(suiteName):
                break
        else:
            # Land here if no "break" was issued
            timeoutMap[suiteName] = ev

    return results.values() + timeoutMap.values()










#=== The tests describe how a running test (or subset test) should map on a previous test run

class EventStub(dict):
    'A very relaxed event stub'
    def __getattr__(self, item): return self[item]
    def getProperties(self):     return self


def test_can_convert_result_event_to_tuple():
    ev = EventStub({ "type": "result", "result": "PASSED", "pageName": "ExampleSuite.SomePage1", "other": "xaxis", "yet-another": 2})
    t = events_to_tuples([ev], "other")
    assert ('result', 'xaxis', 'PASSED', 'ExampleSuite.SomePage1')


def test_can_convert_timeout_event_to_tuple():
    ev = EventStub({ "type": "jobStatus", "reason": "timeout", "ntests": 3, "suiteName": "ExampleSuite", "other": "xaxis", "yet-another": 2})
    t = events_to_tuples([ev], "other")
    assert ('timeout', 'xaxis', 3, 'ExampleSuite')


def test_should_shadow_old_results_by_new_ones():
    freshEvents = [
        ('result', None, 'PASSED', 'ExampleSuite.SomePage1'),
        ('result', None, 'FAILED', 'ExampleSuite.SomePage2')
    ]
    oldEvents = [
        ('result', None, 'FAILED', 'ExampleSuite.SomePage1'),
        ('result', None, 'PASSED', 'ExampleSuite.SomePage2')
    ]

    data = overlay_data(freshEvents, oldEvents)

    assert len(data) == 2, data

    assert ('result', None, 'PASSED', 'ExampleSuite.SomePage1') in data
    assert ('result', None, 'FAILED', 'ExampleSuite.SomePage2') in data


def test_timeout_should_override_previous_test_results():
    freshEvents = [
        ('timeout', None, '2', 'ExampleSuite')
    ]
    oldEvents = [
        ('result', None, 'FAILED', 'ExampleSuite.SomePage1'),
        ('result', None, 'PASSED', 'ExampleSuite.SomePage2'),
        ('result', None, 'PASSED', 'AnotherExampleSuite.SomePage1')
    ]

    data = overlay_data(freshEvents, oldEvents)

    assert len(data) == 2, data

    assert ('timeout', None, '2', 'ExampleSuite') in data
    assert ('result', None, 'PASSED', 'AnotherExampleSuite.SomePage1') in data


def test_test_results_should_override_previous_timeout():
    freshEvents = [
        ('result', None, 'PASSED', 'ExampleSuite.SomePage1'),
        ('result', None, 'FAILED', 'ExampleSuite.SomePage2')
    ]
    oldEvents = [
        ('timeout', None, '2', 'ExampleSuite'),
        ('result', None, 'PASSED', 'AnotherExampleSuite.SomePage1')
    ]

    data = overlay_data(freshEvents, oldEvents)

    assert len(data) == 3, data

    assert ('result', None, 'PASSED', 'ExampleSuite.SomePage1') in data
    assert ('result', None, 'FAILED', 'ExampleSuite.SomePage2') in data
    assert ('result', None, 'PASSED', 'AnotherExampleSuite.SomePage1') in data

def test_timeout_without_new_results_should_still_be():
    freshEvents = [
        ('result', None, 'PASSED', 'ExampleSuite.SomePage1'),
        ('result', None, 'FAILED', 'ExampleSuite.SomePage2')
    ]
    oldEvents = [
        ('timeout', None, '2', 'ExampleSuite'),
        ('timeout', None, '2', 'AnotherExampleSuite')
    ]

    data = overlay_data(freshEvents, oldEvents)

    assert len(data) == 3, data

    assert ('result', None, 'PASSED', 'ExampleSuite.SomePage1') in data
    assert ('result', None, 'FAILED', 'ExampleSuite.SomePage2') in data
    assert ('timeout', None, '2', 'AnotherExampleSuite') in data

